package com.xebialabs.deployit.core.api.resteasy;

import com.google.common.collect.Lists;
import org.jboss.resteasy.client.core.ClientErrorInterceptor;
import org.jboss.resteasy.plugins.providers.ByteArrayProvider;
import org.jboss.resteasy.plugins.providers.RegisterBuiltin;
import org.jboss.resteasy.plugins.providers.StringTextStar;
import org.jboss.resteasy.plugins.providers.jaxb.*;
import org.jboss.resteasy.plugins.providers.multipart.*;
import org.jboss.resteasy.spi.ResteasyProviderFactory;

import javax.ws.rs.ext.ContextResolver;
import javax.ws.rs.ext.ExceptionMapper;
import javax.ws.rs.ext.MessageBodyReader;
import javax.ws.rs.ext.MessageBodyWriter;
import java.util.Arrays;
import java.util.List;

/**
 */
public class ResteasyHelper {

	public static class ResteasyProviderFactoryBuilder {
		private ResteasyProviderFactory factory;

        private List<Class<? extends MessageBodyReader<?>>> readers = Lists.newArrayList();
		private List<Class<? extends MessageBodyWriter<?>>> writers = Lists.newArrayList();
		private List<ClientErrorInterceptor> errorInterceptors = Lists.newArrayList();
		private List<Class<? extends ContextResolver<?>>> contextResolvers = Lists.newArrayList();
		private List<Class<? extends ExceptionMapper<?>>> exceptionMappers = Lists.newArrayList();

		private ResteasyProviderFactoryBuilder(final ResteasyProviderFactory factory) {
			this.factory = factory;
		}

		private void registerWriters() {
			for (Class<? extends MessageBodyWriter<?>> writer : writers) {
				factory.addMessageBodyWriter(writer);
			}
		}

		private void registerReaders() {
			for (Class<? extends MessageBodyReader<?>> reader : readers) {
				factory.addMessageBodyReader(reader);
			}
		}

		private void registerClientErrorInterceptors() {
			for (ClientErrorInterceptor interceptor : errorInterceptors) {
				factory.addClientErrorInterceptor(interceptor);
			}
		}

		private void registerContextResolvers() {
			for (Class<? extends ContextResolver<?>> contextResolver : contextResolvers) {
				factory.addContextResolver(contextResolver);
			}
		}

		private void registerExceptionMappers() {
			for (Class<? extends ExceptionMapper<?>> exceptionMapper : exceptionMappers) {
				factory.addExceptionMapper(exceptionMapper);
			}
		}

		public ResteasyProviderFactoryBuilder addReader(Class<? extends MessageBodyReader<?>>... reader) {
			readers.addAll(Arrays.asList(reader));
			return this;
		}

		public ResteasyProviderFactoryBuilder addWriter(Class<? extends MessageBodyWriter<?>>... writer) {
			writers.addAll(Arrays.asList(writer));
			return this;
		}

		public ResteasyProviderFactoryBuilder addContextResolver(final Class<? extends ContextResolver<?>>... contextFinderClasses) {
			contextResolvers.addAll(Arrays.asList(contextFinderClasses));
			return this;
		}

		public ResteasyProviderFactoryBuilder addClientErrorInterceptor(ClientErrorInterceptor... interceptor) {
			errorInterceptors.addAll(Arrays.asList(interceptor));
			return this;
		}

		public ResteasyProviderFactoryBuilder addExceptionMapper(Class<? extends ExceptionMapper<?>>... exceptionMapper) {
			exceptionMappers.addAll(Arrays.asList(exceptionMapper));
			return this;
		}

		public void build() {
			registerReaders();
			registerWriters();
			registerClientErrorInterceptors();
			registerContextResolvers();
			registerExceptionMappers();
			try {
				RegisterBuiltin.registerProviders(factory);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}

		private ResteasyProviderFactoryBuilder addGenericProviders() {
			addReaderWriter(StringTextStar.class);
			addReaderWriter(ByteArrayProvider.class);
			return this;
		}
		
		@SuppressWarnings("unchecked")
        private <T extends MessageBodyReader<?> & MessageBodyWriter<?>> ResteasyProviderFactoryBuilder addReaderWriter(Class<T> readerWriter) {
			addReader(readerWriter);
			addWriter(readerWriter);
			return this;
		}

		@SuppressWarnings("unchecked")
        private ResteasyProviderFactoryBuilder addMultipartProviders() {
			addReader(MultipartReader.class, MultipartRelatedReader.class, MultipartFormDataReader.class, ListMultipartReader.class, MapMultipartFormDataReader.class, MultipartFormAnnotationReader.class, MimeMultipartProvider.class, XopWithMultipartRelatedReader.class);
			addWriter(MultipartWriter.class, MultipartRelatedWriter.class, MultipartFormDataWriter.class, ListMultipartWriter.class, MapMultipartFormDataWriter.class, MultipartFormAnnotationWriter.class, MimeMultipartProvider.class, XopWithMultipartRelatedWriter.class);
			return this;
		}

		@SuppressWarnings("unchecked")
        private ResteasyProviderFactoryBuilder addJaxbProviders() {
			addReader(JAXBXmlRootElementProvider.class, JAXBXmlTypeProvider.class, JAXBXmlSeeAlsoProvider.class, JAXBElementProvider.class);
			addWriter(JAXBXmlRootElementProvider.class, JAXBXmlTypeProvider.class, JAXBXmlSeeAlsoProvider.class, JAXBElementProvider.class);
			addContextResolver(XmlJAXBContextFinder.class);
			return this;
		}
		
		@SuppressWarnings("unchecked")
		private ResteasyProviderFactoryBuilder addCsvProviders() {
			addWriter(CsvMessageProvider.class);
			return this;
		}
	}

	public static ResteasyProviderFactoryBuilder buildDefaultResteasyProviderFactory() {
		ResteasyProviderFactory rpf = ResteasyProviderFactory.getInstance();
		final ResteasyProviderFactoryBuilder builder = new ResteasyProviderFactoryBuilder(rpf);
		builder.addCsvProviders().addGenericProviders().addJaxbProviders().addMultipartProviders();
		builder.addClientErrorInterceptor(new InternalServerErrorClientResponseInterceptor());

		return builder;
	}

}
