package com.xebialabs.deployit.core.api.dto;

import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Multimap;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement(name="security-settings")
public class PrincipalPermissions extends AbstractDTO {

	@XmlAttribute
	private String username;

	@XmlElement
	private List<PrincipalPermission> permissions = newArrayList();

	public PrincipalPermissions() {
	}

	public PrincipalPermissions(final String userName, final Multimap<String, String> principalPermissions) {
		this.username = userName;
		if (principalPermissions != null) {
			convert(principalPermissions);
		}
	}

	private void convert(Multimap<String, String> principalPermissions) {
		for (String node : principalPermissions.keySet()) {
			permissions.add(new PrincipalPermission(node, newArrayList(principalPermissions.get(node))));
		}
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(final String userName) {
		this.username = userName;
	}

	public boolean hasPermissions() {
		return !this.permissions.isEmpty();
	}

	public boolean hasPermission(final String permission) {
		return !filter(permissions, new Predicate<PrincipalPermission>() {
			public boolean apply(PrincipalPermission input) {
				return Strings.isNullOrEmpty(input.node) && input.permissions.contains(permission);
			}
		}).isEmpty();
	}

	public boolean hasPermission(final String node, final String permission) {
		return !filter(permissions, new Predicate<PrincipalPermission>() {
			public boolean apply(PrincipalPermission input) {
				return ((node == input.node) || (node != null && node.equals(input.node))) && input.permissions.contains(permission);
			}
		}).isEmpty();
	}

	@Override
	public String toString() {
		StringBuilder permissionString = new StringBuilder();
		if ("admin".equals(username)) {
			permissionString.append("User admin has all permissions.\n");
		} else if ((permissions == null) || permissions.isEmpty()) {
			permissionString.append("No permissions found.\n");
		} else {
			for (PrincipalPermission permissionEntry : permissions) {
				if (permissionEntry.node.equals("")) {
					permissionString.append(format("\nGranted global permissions:\n"));
					appendPermissions(permissionString, permissionEntry.permissions);
				} else {
					permissionString.append(format("\nGranted permissions on node '%s':\n", permissionEntry.node));
					appendPermissions(permissionString, permissionEntry.permissions);
				}
			}
		}
		return permissionString.toString();

	}

	private void appendPermissions(StringBuilder permissionString, List<String> permissions) {
		for (String permission : permissions) {
			permissionString.append(format("\t- %s\n", permission));
		}
	}

	@XmlRootElement
	private static class PrincipalPermission {
		@XmlElement
		private String node;
		
		@XmlElement
		private List<String> permissions = newArrayList();

		private PrincipalPermission(String node, List<String> permissions) {
			this.node = node;
			this.permissions = permissions;
		}

		public PrincipalPermission() {
		}
	}
}
