package com.xebialabs.deployit.plugins

import ai.digital.configuration.central.deploy.db.{DatabaseProperties, MainDatabase}
import com.tqdev.metrics.core.MetricRegistry
import com.tqdev.metrics.jdbc.InstrumentedDataSource
import com.xebialabs.deployit.ServerLaunchOptions
import com.xebialabs.deployit.core.config.SqlConfiguration
import com.xebialabs.deployit.core.config.SqlConfiguration.hikariDataSource
import com.xebialabs.deployit.core.metrics.XldDbMetricsTrackerFactory
import com.xebialabs.deployit.core.sql.{SchemaInfo, SqlDialect}
import com.xebialabs.deployit.plumbing.CurrentVersion
import com.xebialabs.plugin.manager.config.PluginManagerProperties
import com.xebialabs.plugin.manager.config.impl.PluginManagerPropertiesImpl
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.service.LocalPluginService
import com.xebialabs.plugin.manager.startup.{PluginInserter, PluginSynchronizer, PluginUpgrader, SourceOfTruth}
import grizzled.slf4j.Logging
import liquibase.Liquibase
import liquibase.database.jvm.JdbcConnection
import liquibase.resource.ClassLoaderResourceAccessor
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.{Bean, Configuration, Lazy}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.datasource.DataSourceTransactionManager
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.interceptor.TransactionProxyFactoryBean

import java.beans.BeanProperty
import java.util.Properties
import javax.annotation.PostConstruct
import javax.sql.DataSource


/**
 * XldProductConfiguration and LocalPluginService are created in a lazy way because they are not needed in PluginManagerCli use case
 */
@Configuration
class PluginManagerConfiguration extends Logging {


  @Bean
  def serverLaunchOptions() = new ServerLaunchOptions

  @Bean
  @ConfigurationProperties(prefix = "deploy.plugins")
  def pluginManagerProperties: PluginManagerProperties = new PluginManagerPropertiesImpl

  @Bean
  def pluginSynchronizer(repository: SqlPluginRepository, launchOptions: ServerLaunchOptions): PluginSynchronizer = {
    val pluginSource = launchOptions.getPluginSource
    val sourceOfTruth = if (pluginSource != null) SourceOfTruth.withName(pluginSource) else SourceOfTruth.DATABASE
    new PluginSynchronizer(repository, "xl-deploy", sourceOfTruth)
  }

  @Bean
  def sqlPluginRepository(@Qualifier("pluginManagerJdbcTemplate") jdbcTemplate: JdbcTemplate,
                          @Qualifier("pluginManagerTransactionManager") tm: PlatformTransactionManager): TransactionProxyFactoryBean = {
    val target = new SqlPluginRepository(jdbcTemplate)
    val factoryBean = new TransactionProxyFactoryBean()
    factoryBean.setTransactionManager(tm)
    factoryBean.setTarget(target)
    factoryBean.setProxyTargetClass(true)
    val txProperties = new Properties()
    txProperties.setProperty("*", "PROPAGATION_REQUIRED,ISOLATION_READ_COMMITTED")
    factoryBean.setTransactionAttributes(txProperties)
    factoryBean
  }

  @Bean
  @Lazy
  def localPluginService(repository: SqlPluginRepository): LocalPluginService = {
    new LocalPluginService(pluginManagerProperties, "xl-deploy", CurrentVersion.get, repository)
  }

  @BeanProperty
  var database: DatabaseProperties = new DatabaseProperties()

  @Autowired(required = true)
  var mainDatabase: MainDatabase = _

  @Autowired
  var xldDbMetricsTrackerFactory: XldDbMetricsTrackerFactory = _

  @Bean(destroyMethod = "close")
  def pluginManagerDataSource: DataSource = {
    val database = if (this.database.hasConfigured) this.database else mainDatabase.database
    instrumentedDataSource(hikariDataSource(database, "PluginManagerPool", xldDbMetricsTrackerFactory))
  }

  @Bean
  def pluginManagerSchema: SchemaInfo = {
    val dialect = SqlDialect.initializeDialect(pluginManagerDataSource)
    logger.info(s"Detected SQL dialect for plugin manager: $dialect.")
    SchemaInfo(dialect, Option(this.database.dbSchemaName).orElse(None))
  }

  @Bean
  def pluginManagerJdbcTemplate: JdbcTemplate = new JdbcTemplate(pluginManagerDataSource, true)

  @Bean
  def pluginManagerTransactionManager: PlatformTransactionManager =
    new DataSourceTransactionManager(pluginManagerDataSource)

  @Bean
  def pluginInserter(repository: SqlPluginRepository): PluginInserter = new PluginInserter(repository, "xl-deploy")

  @Bean
  def pluginUpgrader(repository: SqlPluginRepository): PluginUpgrader =  new PluginUpgrader(repository, "xl-deploy")

  @PostConstruct
  def initializePluginManagerDatabase(): Unit =
    updateLiquibase("com/xebialabs/plugin/manager/db/db.changelog-plugin-manager.yaml", pluginManagerDataSource, pluginManagerSchema.sqlDialect)

  // TODO duplicated code, this method also exists in SqlConfiguration in deploy-task-engine repo, but it's not public
  private def updateLiquibase(schema: String, dataSource: DataSource, dialect: SqlDialect): Unit = {
    import SqlConfiguration.liquibaseContext
    val connection = dataSource.getConnection
    try {
      val context = liquibaseContext(dialect, connection)
      logger.info(s"Detected liquibase context: $context")
      val liquibase = new Liquibase(
        schema,
        new ClassLoaderResourceAccessor(),
        new JdbcConnection(connection)
      )
      liquibase.update(context, schema)
    } finally {
      connection.close()
    }
  }

  private def instrumentedDataSource(dataSource: DataSource) = new InstrumentedDataSource(dataSource, MetricRegistry.getInstance())

}
