package com.xebialabs.deployit

import ai.digital.configuration.central.deploy.InitialCentralConfigSetup
import ai.digital.deploy.core.common.XldServerPaths._
import ch.qos.logback.classic.{Level, LoggerContext}
import com.xebialabs.deployit.ServerLaunchOptions.parseCommandLine
import com.xebialabs.deployit.configuration.system.SystemValues
import com.xebialabs.deployit.env.ServerActiveProfileRegistry
import com.xebialabs.deployit.log.LogbackHelper.{disableConsoleLoggerIfNeeded, enableJavaUtilLoggingBridge}
import com.xebialabs.deployit.util.{DeployitKeys, PasswordEncrypter}
import com.xebialabs.platform.script.jython.EngineInstance
import com.xebialabs.plugin.Xlp
import org.bouncycastle.jce.provider.BouncyCastleProvider
import org.slf4j.LoggerFactory

import java.nio.file.Files
import java.security.Security
import java.util.Collections
import java.util.Objects.{isNull, nonNull}
import scala.jdk.CollectionConverters._

object ApplicationBooter {

  private val logger = LoggerFactory.getLogger(classOf[DeployitBootstrapper])
  val JAVAX_NET_SSL_TRUST_STORE_PASSWORD = "javax.net.ssl.trustStorePassword"

  var xlDeployServer: XLDeployServer = _

  def setUpTrustStorePassword(): Unit =
    if (isNull(System.getProperty(JAVAX_NET_SSL_TRUST_STORE_PASSWORD))
      && nonNull(ServerConfiguration.getInstance) &&
      nonNull(ServerConfiguration.getInstance.getTrustStorePassword)) {
      System.setProperty(JAVAX_NET_SSL_TRUST_STORE_PASSWORD, ServerConfiguration.getInstance.getTrustStorePassword)
    }

  def boot(args: Array[String]): Unit = {
    bootLoggers()

    bootPluginClassloader()

    val launchOptions = parseCommandLine(args)
    if (launchOptions == null) System.exit(0)

    initialiseJython()

    Security.addProvider(new BouncyCastleProvider)

    bootConfigs(launchOptions)

    PasswordEncrypter.init(DeployitKeys.getPasswordEncryptionKey(launchOptions.getRepositoryKeystorePassword))

    bootSystemProperties(launchOptions)

    if (launchOptions.isDoSetup) {
      System.exit(0)
    }
  }

  def bootPluginManagerCli(args: Array[String]): Unit = {
    val launchOptions = parseCommandLine(args)
    if (launchOptions == null) System.exit(0)

    Security.addProvider(new BouncyCastleProvider)

    bootConfigs(launchOptions)

    PasswordEncrypter.init(DeployitKeys.getPasswordEncryptionKey(launchOptions.getRepositoryKeystorePassword))

    bootSystemProperties(launchOptions)

    if (launchOptions.isDoSetup) {
      System.exit(0)
    }
  }

  private def bootLoggers(): Unit = {
    disableConsoleLoggerIfNeeded("STDOUT")
    enableJavaUtilLoggingBridge()
    setUpDefaultLogLevels()
  }

  private def bootPluginClassloader(): Unit = {
    Files.createDirectories(XL_HOTFIX_PLUGINS_FOLDER.toPath)
    Files.createDirectories(XL_PLUGINS_LOCAL_FOLDER.toPath)
    Files.createDirectories(XL_PLUGINS_OFFICIAL_FOLDER.toPath)
    Xlp.init(XL_PLUGIN_EXTENSION, List(XL_HOTFIX_PLUGINS_FOLDER, XL_PLUGINS_LOCAL_FOLDER, XL_PLUGINS_OFFICIAL_FOLDER).asJava, Collections.singletonList(EXT_FOLDER))
  }

  private def bootConfigs(launchOptions: ServerLaunchOptions): Unit = {
    val serverConfigFile = new ServerConfigFile(DEFAULT_CONFIGURATION_FILE)
    xlDeployServer = new XLDeployServer(launchOptions, serverConfigFile)
    xlDeployServer.prepareConfigs()
    InitialCentralConfigSetup.setup()
  }

  private def bootSystemProperties(launchOptions: ServerLaunchOptions): Unit = {
    val profiles = ServerActiveProfileRegistry.getActiveProfiles(launchOptions, ServerConfiguration.getInstance())
    if (isExternalConfig(launchOptions, ServerConfiguration.getInstance())) {
      SystemValues
        .apply(profiles)
        .applyConfigClientOnlySystemValues()
    } else {
      SystemValues
        .apply(profiles)
        .applyConfigEmbeddedServerSystemValues(profiles)
    }
    if (ServerConfiguration.getInstance() != null) {
      SystemValues.apply(ServerConfiguration.getInstance())
    }
  }

  private def isExternalConfig(launchOptions: ServerLaunchOptions, serverConfiguration: ServerConfiguration): Boolean =
    Option(launchOptions.isExternalConfig())
      .map(Boolean.unbox)
      .orElse(Option(serverConfiguration).map(_.getSpringCloudExternalConfig()))
      .getOrElse(false)

  private def setUpDefaultLogLevels(): Unit = {
    val lc = LoggerFactory.getILoggerFactory.asInstanceOf[LoggerContext]
    lc.getLogger("org.eclipse.jetty.server.session").setLevel(Level.WARN)
  }

  private def initialiseJython(): Unit =
    try {
      logger.info("Starting up Jython engine (might take a while)...")
      EngineInstance.jython.eval("True")
      logger.info("Jython engine loaded [" + EngineInstance.jython.getFactory.getEngineVersion + "]")
    } catch {
      case e: Exception =>
        logger.warn("Exception while initialising jython engine", e)
    }
}
