package com.xebialabs.deployit.event

import java.io.File
import ai.digital.config.server.client.ConfigurationRefresh
import ai.digital.deploy.task.serdes.TaskPekkoSerializer
import com.tqdev.metrics.core.{Gauge, MetricRegistry}
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskExecutionState}
import com.xebialabs.deployit.engine.spi.event.SystemStartedEvent
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import com.xebialabs.deployit.engine.tasker.distribution.versioning.ConfigurationHashProvider
import com.xebialabs.deployit.engine.tasker.log.{StepLogFactory, StepLogRetriever}
import com.xebialabs.deployit.engine.tasker.repository.{ActiveTaskRepository, PendingTaskRepository}
import com.xebialabs.deployit.engine.tasker.{AddressExtension, Archive, TaskExecutionEngine, TaskQueueService}
import com.xebialabs.deployit.jetty.DeployitSpringContextLoaderListener
import com.xebialabs.deployit.local.message.MessageService
import com.xebialabs.deployit.spring.{BeanWrapper, EngineBeanBuilder}
import com.xebialabs.deployit.tasksystem.TaskActorSystem
import com.xebialabs.deployit.{ServerConfiguration, ServerState}
import com.xebialabs.xlplatform.config.{ConfigLoader, ConfigurationHolder}
import com.xebialabs.xlplatform.pekko.Pekko
import com.xebialabs.xlplatform.scheduler.spring.ServiceHolder
import com.xebialabs.xlplatform.settings.shared.TaskerSettings
import org.apache.pekko.actor.ActorSystem
import jakarta.annotation.PreDestroy
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.boot.context.event.ApplicationReadyEvent
import org.springframework.context.event.EventListener
import org.springframework.jms.config.JmsListenerContainerFactory
import org.springframework.jms.listener.DefaultMessageListenerContainer
import org.springframework.stereotype.Component

import scala.concurrent.duration.Duration
import scala.jdk.CollectionConverters._

@Component
class DeployApplicationReadyListener {

  private val logger = LoggerFactory.getLogger(classOf[DeployApplicationReadyListener])

  @Autowired var activeTaskRepository: ActiveTaskRepository = _
  @Autowired var pendingTaskRepository: PendingTaskRepository = _
  @Autowired var taskExecutionWorkerRepository: TaskExecutionWorkerRepository = _
  @Autowired var stepLogFactory: StepLogFactory = _
  @Autowired var stepLogRetriever: StepLogRetriever = _
  @Autowired var configurationHashProvider: ConfigurationHashProvider = _
  @Autowired var taskerSettings: TaskerSettings = _
  @Autowired var taskQueueService: TaskQueueService = _
  @Autowired var archive: Archive = _
  @Autowired var repositoryAdapter: RepositoryFactory = _
  @Autowired var engineBeanBuilder: EngineBeanBuilder = _
  @Autowired var metricRegistry: MetricRegistry = _
  @Autowired var engine: BeanWrapper[TaskExecutionEngine] = _
  @Autowired var serverConfiguration: ServerConfiguration = _

  @Qualifier("localConfigurationRefresh")
  @Autowired var configurationRefresh: ConfigurationRefresh = _

  @Autowired
  @Qualifier("xlJmsListenerContainerFactory")
  var containerFactory: JmsListenerContainerFactory[DefaultMessageListenerContainer] = _

  @Autowired var deployMessageService: MessageService = _

  @Value("#{baseWorkDir}") var baseWorkDir: File = _
  @Value("${deploy.task.in-process-worker:true}") var inProcessTaskEngine: Boolean = _

  private val configBlacklist = List("akka")

  @EventListener(Array(classOf[ApplicationReadyEvent]))
  def onApplicationReady(): Unit = {

    configurationRefresh.refreshCentralConfiguration()
    checkForBlacklistedConfig()

    implicit val system: ActorSystem = TaskActorSystem.actorSystem
    initEngineBeanBuilder(system)
    setSerializer(system)
    ServiceHolder.setTaskExecutionEngine(engine.get())

    addTaskCounters()
    onApplicationEvent()
    printAddress(system)
  }

  def initEngineBeanBuilder(system: ActorSystem): Unit = {
    engineBeanBuilder.initWorkerManager(system, taskExecutionWorkerRepository, configurationHashProvider)

    engineBeanBuilder.initEngine(system, activeTaskRepository, pendingTaskRepository, containerFactory,
      taskExecutionWorkerRepository, stepLogFactory, stepLogRetriever,
      configurationHashProvider, taskerSettings, taskQueueService)

    engineBeanBuilder.initDiscoveryService(system)
  }

  def setSerializer(system: ActorSystem): Unit = {
    if (!inProcessTaskEngine) {
      TaskPekkoSerializer.initExternalWorker(TaskPekkoSerializer.Conf(baseWorkDir), system, createWorkDir = false)
    }
  }

  def addTaskCounters(): Unit = {
    val tasksGroup = "xldeploy.Tasks"
    metricRegistry.set(tasksGroup, "executing", new Gauge() {
      override def measure: Long =
        engine.get().getAllIncompleteTasks(FetchMode.SUMMARY).asScala.count(_.getState.isExecutingSteps)
    })
    metricRegistry.set(tasksGroup, "queued", new Gauge() {
      override def measure: Long =
        engine.get().getAllIncompleteTasks(FetchMode.SUMMARY).asScala.count(_.getState == TaskExecutionState.QUEUED)
    })
  }

  private def printAddress(system: ActorSystem): Unit = {
    if (!system.settings.config.getBoolean("deploy.task.in-process-worker")) {
      val host: String = AddressExtension.hostOf(system)
      if (host.nonEmpty) logger.info("External workers can connect to {}:{}", host, AddressExtension.portOf(system))
    }
    logger.info(deployMessageService.getMessage("server.url", serverConfiguration.getServerUrl))
  }

  private def onApplicationEvent(): Unit = {
    import com.xebialabs.xlplatform.scheduler.EventListener
    new EventListener()
    EventBusHolder.register(ServerState.getInstance)
    DeployitSpringContextLoaderListener.checkCorrectlyInitialized()
    EventBusHolder.publish(new SystemStartedEvent())
    logger.info("XL Deploy has started.")
  }

  // May be this can be removed after 23.3 support end in an assumption that all customers would have migrated from akka to pekko.
  private def checkForBlacklistedConfig(): Unit = {
    val deployConfig: String = ConfigLoader.loadWithDynamic(ConfigurationHolder.get()).toString
    Option(configBlacklist.foreach(bl => {
      if (deployConfig.contains(bl)) {
        logger.debug(deployConfig)
        val message = s"Configuration settings found which contains not supported term: $bl" +
          " See the migration guide for details about fixing your configuration."
        logger.error(message, new IllegalStateException(message))
        System.exit(1)
      }
    }))
  }

  @PreDestroy
  def destroy(): Unit = {
    implicit val system: ActorSystem = TaskActorSystem.actorSystem
    Pekko.terminate(system, Duration.Inf)
  }
}
