/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.setup;

import ai.digital.deploy.core.common.XldServerPaths;
import com.xebialabs.deployit.ServerConfigFile;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.ServerLaunchOptions;
import com.xebialabs.deployit.booter.local.LocalBooter;
import com.xebialabs.deployit.booter.local.utils.Closeables;
import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.util.DeployitKeyStore;
import com.xebialabs.deployit.util.DeployitKeys;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.overthere.util.OverthereUtils;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.LoaderOptions;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;

import javax.crypto.SecretKey;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.*;

import static com.xebialabs.deployit.ServerConfiguration.*;
import static com.xebialabs.deployit.setup.SetupHelperMethods.*;
import static com.xebialabs.overthere.util.OverthereUtils.checkState;

public class Setup {

    private static final Logger logger = LoggerFactory.getLogger(Setup.class);

    public static final String TRUSTSTORE_PATH = "What is the path to the truststore?";
    public static final String KEYSTORE_PATH = "What is the path to the keystore?";

    public static final String JDBC_URL = "What is the JDBC URL to use?";
    public static final String JDBC_USER_NAME = "What is the JDBC username?";
    public static final String JDBC_DRIVER_NAME = "What is the JDBC driver name? ";

    private final String serverName;
    private final ServerConfigFile serverConfigFile;
    private final ServerLaunchOptions launchOptions;
    private Map<String, String> newDbConfigs;

    protected ServerConfiguration configuration;

    // Wizard options
    private final boolean askServerUrl = false;

    // Setup options
    private boolean generateKeyStore;
    private boolean generatePasswordEncryptionKey;
    private String passwordEncryptionKeyPassword;

    private final File repositoryFile = new File(XldServerPaths.CENTRAL_CONFIG_FOLDER(), "deploy-repository.yaml");

    public Setup(String serverName, ServerConfigFile serverConfigFile, ServerLaunchOptions deployitOptions) {
        this(serverName, serverConfigFile, deployitOptions, new ServerConfiguration());
    }

    public Setup(String serverName, ServerConfigFile serverConfigFile, ServerLaunchOptions deployitOptions, ServerConfiguration defaultConfiguration) {
        this.serverName = serverName;
        this.serverConfigFile = serverConfigFile;
        this.launchOptions = deployitOptions;
        this.configuration = defaultConfiguration;
    }

    public void run() {
        logger.info("About to start XL Deploy");
        LocalBooter.boot(XldServerPaths.TYPE_DEFAULTS());
        logger.info("The DescriptorRegistry has been booted");
        final String previousInstallation = launchOptions.getPreviousInstallation();
        if (Strings.isNotBlank(previousInstallation)) {
            runUpgrade(previousInstallation);
        } else if (launchOptions.isReinitialize()) {
            reinitialize();
        } else {
            runComplete();
        }
    }

    private void runUpgrade(final String previousInstallation) {
        if (!launchOptions.isDoSetup()) {
            throw new RuntimeException("Option `-previous-installation` is only supported in setup mode (with `-setup` option)");
        } else {
            System.out.println("\nWelcome to the " + serverName + " setup.");
            System.out.println("You can always exit by typing '" + EXITSETUP_KEYWORD + "'.");
            final UpgradeHelper upgradeHelper = new UpgradeHelper(previousInstallation, launchOptions.isForceUpgrades(), new FileCopier());
            SetupConfiguration config = readSetupConfigurationProperties();

            final List<String> filesToMention = Collections.emptyList();
            final List<String> dirs = List.of("centralConfiguration", "ext");

            upgradeHelper.copyData(config.getConfigFiles().getMandatory(), dirs,
                    filesToMention, config.getConfigFiles().getOptional());

            serverConfigFile.appendConfiguration(Collections.singletonMap(
                    KEY_SPRING_CLOUD_ENCRYPT_KEY, configuration.generateCloudKey()));

            try {
                File centralConfigFile = XldServerPaths.DEFAULT_CENTRAL_CONFIG_VERSION_FILE();
                if (!previousInstallation.isEmpty() && !centralConfigFile.exists()) {
                    FileUtils.writeStringToFile(centralConfigFile, "central.configuration.version=10.0.0", StandardCharsets.UTF_8);
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private SetupConfiguration readSetupConfigurationProperties() {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        InputStream inputStream = loader.getResourceAsStream("setup.yaml");
        return new Yaml(new Constructor(SetupConfiguration.class, new LoaderOptions())).
                loadAs(inputStream, SetupConfiguration.class);
    }

    private void reinitialize() {
        if (!launchOptions.isForce()) {
            System.out.println("The -reinitialize will only work with a default repository setup; Not when you've configured " + serverName + "  to integrate with a database.");
            System.out.println("Are you sure you want to continue (yes or no)?");
            if (!getBooleanResponse(false)) {
                return;
            }
        }

        // Load password encryption keys
        String repositoryKeystorePassword = launchOptions.getRepositoryKeystorePassword();
        SecretKey passwordEncryptionKey;
        if (null != repositoryKeystorePassword) {
            DeployitKeyStore.generateRandomKeyStore(new File("conf"), repositoryKeystorePassword);
            passwordEncryptionKey = DeployitKeyStore.getPasswordEncryptionKey();
        } else {
            passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(repositoryKeystorePassword);
        }
        PasswordEncrypter.init(passwordEncryptionKey);

        useDefaultValues(launchOptions.getDefaultsFile());
        if (configuration.getHttpPort() == 0) {
            configuration.setHttpPort(defaultHttpPort());
        }
        configuration.setSpringCloudEnabled(true);
        serverConfigFile.writeConfiguration(configuration);
        configuration = serverConfigFile.loadConfig(true, true, true);
        deleteTaskRecoveryFile();
    }

    private void runComplete() {
        System.out.println("\nWelcome to the " + serverName + " setup.");
        System.out.println("You can always exit by typing '" + EXITSETUP_KEYWORD + "'.");
        System.out.println("To re-run this setup and make changes to the " + serverName +
                " configuration you can run run.cmd -setup on Windows or run.sh -setup on Unix.");

        boolean editingExistingConfiguration = false;
        if (serverConfigFile.exists()) {
            if (askDoYouWantToEditTheExistingConfiguration()) {
                SecretKey passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(launchOptions.getRepositoryKeystorePassword());
                PasswordEncrypter.init(passwordEncryptionKey);
                serverConfigFile.readIntoConfiguration(configuration);
                editingExistingConfiguration = true;
            }
        }

        if (askToUseSimpleSetup()) {
            if (!editingExistingConfiguration) {
                configuration.setHttpPort(DEFAULT_HTTP_PORT);
                askAdminPassword();
            } else {
                upgradeConfigurationWithDefaultValues();
            }
            askToCreatePasswordEncryptionKey();
            askDatabaseConfiguration();
        } else {
            if (!editingExistingConfiguration) {
                askAdminPassword();
            }

            if (askToEnableSsl()) {
                if (askToGenerateKeys()) {
                    setDefaultKeyStoreSettings();
                } else {
                    askKeyStoreSettings();
                }

                if (askToEnableMutualSsl()) {
                    askTrustStoreSettings();
                }
            }

            askHttpBindAddressForJetty();
            askHttpPortForJetty();
            askWebContextRoot();
            askServerUrl();
            askMinumAmountOfThreads();
            askMaximumAmountOfThreads();

            askToCreatePasswordEncryptionKey();
            askDatabaseConfiguration();
        }

        if (reviewAndConfirm()) {
            SecretKey passwordEncryptionKey;

            updateRepositoryConfig();

            if (generatePasswordEncryptionKey) {
                DeployitKeyStore.generateRandomKeyStore(new File("conf"), passwordEncryptionKeyPassword);
                passwordEncryptionKey = DeployitKeyStore.getPasswordEncryptionKey();
            } else {
                // Default key store
                passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(launchOptions.getRepositoryKeystorePassword());
            }
            PasswordEncrypter.init(passwordEncryptionKey);

            System.out.println("Saving to " + serverConfigFile);
            configuration.setSpringCloudEnabled(true);
            serverConfigFile.writeConfiguration(configuration);
            System.out.println("Configuration saved.");

            if (generateKeyStore) {
                generateKeyStore();
            }

            // Load the new configuration
            configuration = serverConfigFile.loadConfig(true, true, true);

            System.out.println("You can now start your " + serverName + " by executing the command run.cmd on Windows or run.sh on Unix.");
            System.out.println("Note: If your " + serverName + " is running please restart it.");
            System.out.println("Finished setup.");
        } else {
            System.out.println("Aborting setup.");
        }
    }

    private void updateRepositoryConfig() {
        try {
            Map<String, Map<String, Map<String, String>>> configToDump =
                    Collections.singletonMap("xl.repository",
                            Collections.singletonMap("database", newDbConfigs));

            if (newDbConfigs != null && !newDbConfigs.isEmpty()) {
                String content = createYamlParser().dump(configToDump);
                Files.write(Paths.get(repositoryFile.getPath()), content.getBytes(StandardCharsets.UTF_8));
            }
        } catch (IOException e) {
           logger.error("Error occurred", e);
        }
    }

    protected void useDefaultValues(String defaultsFile) {
        configuration.setDefaults();

        if (defaultsFile != null) {
            File file = new File(defaultsFile);
            checkState(file.exists(), "The given file [%s] with defaults does not exist.", defaultsFile);
            Properties properties = new Properties();
            FileInputStream inStream = null;
            try {
                inStream = new FileInputStream(file);
                properties.load(inStream);
            } catch (IOException e) {
                throw new RuntimeException("Cannot load [" + file + "] to read defaults from.");
            } finally {
                Closeables.closeQuietly(inStream);
            }
            configuration.load(properties);
        }
    }

    private void askAdminPassword() {
        printEmptyLine();
        System.out.println("Please enter the admin password you wish to use for " + serverName);

        String password = getConfirmedPassword();

        if (Strings.isEmpty(password)) {
            System.out.println("Using default admin password.");
            configuration.setAdminPassword(ServerConfiguration.DEFAULT_ADMIN_PASSWORD);
        } else {
            configuration.setAdminPassword(password);
        }
    }

    private void askWebContextRoot() {
        printEmptyLine();
        System.out.println("Enter the web context root where " + serverName + " will run");
        String contextRoot = getStringResponse(ServerConfiguration.DEFAULT_WEB_CONTEXT_ROOT);
        if (!contextRoot.startsWith("/")) {
            contextRoot = "/" + contextRoot;
        }
        configuration.setWebContextRoot(contextRoot);
    }

    private void askServerUrl() {
        if (!askServerUrl) {
            return;
        }

        printEmptyLine();
        System.out.println("Enter the public URL to access " + serverName);
        String derivedUrl = configuration.getDerivedServerUrl();
        String serverUrl = getStringResponse(derivedUrl);

        // Only store if different
        if (!derivedUrl.equals(serverUrl)) {
            configuration.setServerUrl(serverUrl);
        }
    }

    private void upgradeConfigurationWithDefaultValues() {
        if (configuration.getMinThreads() == 0) {
            configuration.setMinThreads(ServerConfiguration.DEFAULT_MIN_THREADS);
        }
        if (configuration.getMaxThreads() == 0) {
            configuration.setMaxThreads(ServerConfiguration.DEFAULT_MAX_THREADS);
        }
        if (configuration.getHttpPort() == 0) {
            configuration.setHttpPort(DEFAULT_HTTP_PORT);
        }
    }

    private void askMinumAmountOfThreads() {
        printEmptyLine();
        System.out.println("Enter the minimum number of threads the HTTP server should use (recommended: 3 per client, so 30 for 10 concurrent users)");
        int suppliedMinThreads = (configuration.getMinThreads() == 0) ? ServerConfiguration.DEFAULT_MIN_THREADS : configuration.getMinThreads();
        configuration.setMinThreads(getValidIntegerResponse(suppliedMinThreads));
    }

    private void askMaximumAmountOfThreads() {
        printEmptyLine();
        System.out.println("Enter the maximum number of threads the HTTP server should use (recommended :3 per client, so 150 for 50 concurrent users)");
        int suppliedMaxThreads = (configuration.getMaxThreads() == 0) ? ServerConfiguration.DEFAULT_MAX_THREADS : configuration.getMaxThreads();
        configuration.setMaxThreads(getValidIntegerResponse(suppliedMaxThreads));
    }

    private static boolean askToUseSimpleSetup() {
        printEmptyLine();
        System.out.println("Do you want to use the simple setup?");
        System.out.println("Default values are used for all properties. To make changes to the default properties, please answer no.");
        System.out.println("Options are yes or no.");
        return getBooleanResponse(true);
    }

    private boolean askToEnableSsl() {
        printEmptyLine();
        System.out.println("Would you like to enable SSL?");
        System.out.println("Options are yes or no.");
        boolean suppliedSsl = configuration.isNewConfiguration() || configuration.isSsl();
        configuration.setSsl(getBooleanResponse(suppliedSsl));
        return configuration.isSsl();
    }

    private boolean askToGenerateKeys() {
        printEmptyLine();
        System.out.println("Would you like to generate a keystore with a self-signed certificate for you?");
        System.out.println("N.B.: Self-signed certificates do not work correctly with some versions of the Flash Player and some browsers!");
        System.out.println("Options are yes or no.");
        boolean suppliedGenerateKeys = configuration.isNewConfiguration() || generateKeyStore;
        generateKeyStore = getBooleanResponse(suppliedGenerateKeys);
        return generateKeyStore;
    }

    private void setDefaultKeyStoreSettings() {
        configuration.setKeyStorePath("conf/keystore.p12");
        configuration.setKeyStorePassword("keystoresecret");
        configuration.setKeyStoreKeyPassword("keystoresecret");
        configuration.setKeyStoreType("pkcs12");
    }

    private void askKeyStoreSettings() {
        printEmptyLine();
        System.out.println(KEYSTORE_PATH);
        String suppliedKeyStorePath = configuration.isNewConfiguration() ? "" : configuration.getKeyStorePath();
        configuration.setKeyStorePath(!StringUtils.hasText(suppliedKeyStorePath) ? getStringResponseWithRetry(KEYSTORE_PATH) : getStringResponse(suppliedKeyStorePath));

        printEmptyLine();
        System.out.println("What is the type of the keystore?");
        System.out.println("Options are jks, pkcs12, or other keystore types supported on the JVM.");
        String suppliedKeyStoreType = configuration.isNewConfiguration() ? "jks" : configuration.getKeyStorePath();
        configuration.setKeyStoreType(getStringResponse(suppliedKeyStoreType));

        printEmptyLine();
        System.out.println("What is the password to the keystore?");
        configuration.setKeyStorePassword(getPasswordResponse("Password: "));

        printEmptyLine();
        System.out.println("What is the password to the key in the keystore?");
        configuration.setKeyStoreKeyPassword(getPasswordResponse("Password: "));
    }

    private boolean askToEnableMutualSsl() {
        printEmptyLine();
        System.out.println("Would you like to enable mutual SSL?");
        System.out.println("Options are yes or no.");
        boolean suppliedMutualSsl = configuration.isNewConfiguration() || configuration.isMutualSsl();
        configuration.setMutualSsl(getBooleanResponse(suppliedMutualSsl));
        return configuration.isMutualSsl();
    }

    private void askTrustStoreSettings() {
        printEmptyLine();
        System.out.println(TRUSTSTORE_PATH);
        String suppliedTrustStorePath = configuration.isNewConfiguration() ? "" : configuration.getTrustStorePath();
        configuration.setTrustStorePath(!StringUtils.hasText(suppliedTrustStorePath) ? getStringResponseWithRetry(TRUSTSTORE_PATH) : getStringResponse(suppliedTrustStorePath));

        printEmptyLine();
        System.out.println("What is the password to the truststore?");
        configuration.setTrustStorePassword(getPasswordResponse("Password: "));
    }

    private void askHttpBindAddressForJetty() {
        printEmptyLine();
        System.out.println("What http bind address would you like the server to listen on?");
        String defaultAddress = Strings.isEmpty(configuration.getHttpBindAddress()) ? DEFAULT_HTTP_BIND_ADDRESS : configuration.getHttpBindAddress();
        configuration.setHttpBindAddress(getStringResponse(defaultAddress));
    }

    private void askHttpPortForJetty() {
        printEmptyLine();
        System.out.println("What http port number would you like the server to listen on?");
        int suppliedPort = (configuration.getHttpPort() == 0) ? defaultHttpPort() : configuration.getHttpPort();
        configuration.setHttpPort(getValidIntegerResponse(suppliedPort));
    }

    private int defaultHttpPort() {
        return configuration.isSsl() ? DEFAULT_HTTPS_PORT : DEFAULT_HTTP_PORT;
    }

    private boolean askDoYouWantToEditTheExistingConfiguration() {
        boolean useOldConfig;
        printEmptyLine();
        System.out.println("An existing " + serverName + " configuration was found. Do you want to edit it?");
        System.out.println("Options are yes or no. (selecting no will create an empty configuration)");
        useOldConfig = getBooleanResponse(true);
        if (useOldConfig)
            System.out.println("Editing the exisiting configuration.");
        else
            System.out.println("Starting configuration from scratch.");
        return useOldConfig;
    }

    private void askToCreatePasswordEncryptionKey() {
        printEmptyLine();
        System.out.println("The password encryption key protects the passwords stored in the repository. ");
        System.out.println("Do you want to generate a new password encryption key?");
        System.out.println("Options are yes or no.");

        generatePasswordEncryptionKey = getBooleanResponse(true);
        if (generatePasswordEncryptionKey) {
            System.out.println("The password encryption key is optionally secured by a password.");
            System.out.println("Please enter the password you wish to use. (Use an empty password to avoid a password prompt when starting " + serverName + ".)");
            passwordEncryptionKeyPassword = getConfirmedPassword();
        }
    }

    private void askDatabaseConfiguration() {
        printEmptyLine();
        System.out.print("Do you want to configure the database connection in this wizard? ");
        if (getBooleanResponse(false)) {
            newDbConfigs = new HashMap<>();
            System.out.print(JDBC_URL);
            String jdbcUrl = getStringResponseWithRetry(JDBC_URL);
            newDbConfigs.put("db-url", jdbcUrl);

            String driverFound = findJdbcDriver(jdbcUrl);
            System.out.print(JDBC_DRIVER_NAME);
            newDbConfigs.put("db-driver-classname", !StringUtils.hasText(driverFound) ? getStringResponseWithRetry(JDBC_DRIVER_NAME) : getStringResponse(driverFound));

            System.out.print(JDBC_USER_NAME);
            newDbConfigs.put("db-username", getStringResponseWithRetry(JDBC_USER_NAME));

            System.out.print("What is the JDBC password? ");
            newDbConfigs.put("db-password", getPasswordResponse(": "));
        }
    }

    private Yaml createYamlParser() {
        DumperOptions options = new DumperOptions();
        options.setPrettyFlow(true);
        options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);
        return new Yaml(options);
    }

    private String findJdbcDriver(String jdbcUrl) {
        try {
            return DriverManager.getDriver(jdbcUrl).getClass().getName();
        } catch (SQLException e) {
            System.out.println("No registered JDBC driver could be found for '" + jdbcUrl + "'.");
            System.out.println("You can continue setup, but server startup will fail unless you provide the JDBC driver jar.");
            return null;
        }
    }

    private boolean reviewAndConfirm() {
        printEmptyLine();
        System.out.println("Do you agree with the following settings for " + serverName + " and would you like to save them?");
        System.out.println("Changes will be saved in " + serverConfigFile.getName());

        System.out.println("\tSSL will be " + (configuration.isSsl() ? "enabled" : "disabled"));
        if (configuration.isSsl()) {
            if (generateKeyStore) {
                System.out.println("\tKeystore will be generated");
            } else {
                System.out.println("\tKeystore path is " + configuration.getKeyStorePath());
            }
        }
        System.out.println("\tHTTP bind address is " + configuration.getHttpBindAddress());
        System.out.println("\tHTTP port is " + configuration.getHttpPort());
        System.out.println("\tContext root is " + configuration.getWebContextRoot());
        if (askServerUrl) {
            System.out.println("\tPublic Server URL is " + configuration.getServerUrl());
        }
        System.out.println("\tHTTP server will use a minimum of " + configuration.getMinThreads() + " and a maximum of " + configuration.getMaxThreads() + " threads");

        if (launchOptions.isReinitialize()) {
            System.out.println("\tTask recovery file will be deleted");
        }
        if (generatePasswordEncryptionKey) {
            System.out.println("\tPassword encryption key will be generated.");
        }

        if (configuration.isSpringCloudEnabled() && configuration.getSpringCloudUri() != null) {
            System.out.println("\tConfig Server - URI: " + configuration.getSpringCloudUri());

            System.out.println("\tConfig Server - Retry initial interval: " + configuration.getSpringCloudRetryInitialInterval());
            System.out.println("\tConfig Server - Retry max attempts: " + configuration.getSpringCloudRetryMaxAttempts());
            System.out.println("\tConfig Server - Retry max interval: " + configuration.getSpringCloudRetryMaxInterval());
            System.out.println("\tConfig Server - Retry multiplier: " + configuration.getSpringCloudRetryMultiplier());
        }

        if (newDbConfigs != null) {
            System.out.println("\nAnd database changes to " + repositoryFile.getAbsolutePath());

            String dbUrl = newDbConfigs.get("db-url");
            if (dbUrl != null) {
                System.out.println("\tDatabase connection - URL: " + dbUrl);
            }

            String dbDriverClassname = newDbConfigs.get("db-driver-classname");
            if (dbDriverClassname != null) {
                System.out.println("\tDatabase connection - Driver: " + dbDriverClassname);
            }

            String dbUsername = newDbConfigs.get("db-username");
            if (dbUsername != null) {
                System.out.println("\tDatabase connection - Username: " + dbUsername);
            }
        }

        return getBooleanResponse(true);
    }

    private void generateKeyStore() {
        String keyAlgorithm = "RSA";
        String keyStoreKeyAlias = "jetty";
        String dname = "CN=localhost,O=" + serverName + ",C=NL";

        try {
            File keyStoreFile = new File(configuration.getKeyStorePath());
            if (keyStoreFile.exists()) {
                if (!keyStoreFile.delete()) {
                    System.out.println("Existing keystore " + configuration.getKeyStorePath() + " has not been deleted.");
                } else {
                    System.out.println("Existing keystore " + configuration.getKeyStorePath() + " deleted.");
                }
            }
            System.out.println("Generating keystore...");
            String[] keytoolArgs = new String[]{"keytool",
                    "-genkey",
                    "-keyalg", keyAlgorithm,
                    "-keystore", configuration.getKeyStorePath(),
                    "-storepass", configuration.getKeyStorePassword(),
                    "-alias", keyStoreKeyAlias,
                    "-keypass", configuration.getKeyStoreKeyPassword(),
                    "-validity", "366",
                    "-dname", dname,
                    "-deststoretype", "pkcs12"};
            Process p = Runtime.getRuntime().exec(keytoolArgs);

            Thread outputThread = new Thread(new InputStreamToOutputStream(p.getInputStream(), System.out));
            outputThread.start();
            Thread errorThread = new Thread(new InputStreamToOutputStream(p.getErrorStream(), System.err));
            errorThread.start();

            int exitValue;
            try {
                exitValue = p.waitFor();
            } catch (InterruptedException exc) {
                throw new IOException(exc.toString());
            }
            if (exitValue != 0) {
                throw new IOException("keytool exited with status code " + exitValue);
            }
            System.out.println("Keystore generated.");
        } catch (IOException exc) {
            System.err.println("WARNING: Could not generate keystore " + configuration.getKeyStorePath() + ": " + exc);
        }
    }

    private static class InputStreamToOutputStream implements Runnable {
        private final InputStream in;
        private final OutputStream out;

        public InputStreamToOutputStream(InputStream in, OutputStream out) {
            this.in = in;
            this.out = out;
        }

        public void run() {
            OverthereUtils.write(in, out);
        }
    }

    protected static void deleteTaskRecoveryFile() {
        File taskRecoveryFile = new File("recovery.dat");
        if (taskRecoveryFile.exists()) {
            if (taskRecoveryFile.delete()) {
                System.out.println("Task recovery file deleted");
            } else {
                System.err.println("Couldn't delete the task recovery file");
            }
        }
    }

}
