package com.xebialabs.deployit.server.jetty;

import ai.digital.configuration.central.deploy.ClientProperties;
import ai.digital.configuration.central.deploy.ServerSideProperties;
import ai.digital.configuration.central.deploy.server.GzipProperties;
import com.google.common.base.Charsets;
import com.tqdev.metrics.core.MetricRegistry;
import com.xebialabs.deployit.*;
import com.xebialabs.deployit.config.SameSiteHelper;
import com.xebialabs.deployit.errors.LicenseMissingErrorHandler;
import com.xebialabs.deployit.jetty.MaintenanceModeFilter;
import com.xebialabs.deployit.jetty.NoOptionsFilter;
import com.xebialabs.deployit.jetty.RequestHeadersEncodedAsParametersFilter;
import com.xebialabs.deployit.server.jetty.metrics.CustomInstrumentedHandler;
import com.xebialabs.deployit.server.jetty.metrics.CustomMeasureRequestPathFilter;
import com.xebialabs.xlplatform.endpoints.servlet.FlexPekkoStreamServlet;
import com.xebialabs.xlplatform.endpoints.servlet.PekkoStreamServletInitializer;
import jakarta.servlet.DispatcherType;
import jakarta.servlet.SessionTrackingMode;
import jakarta.ws.rs.core.MediaType;
import org.eclipse.jetty.ee10.servlet.DefaultServlet;
import org.eclipse.jetty.ee10.servlet.FilterHolder;
import org.eclipse.jetty.ee10.servlet.ServletHolder;
import org.eclipse.jetty.ee10.servlet.SessionHandler;
import org.eclipse.jetty.ee10.webapp.AbstractConfiguration;
import org.eclipse.jetty.ee10.webapp.Configuration;
import org.eclipse.jetty.ee10.webapp.WebAppContext;
import org.eclipse.jetty.http.HttpCookie;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.server.handler.gzip.GzipHandler;
import org.eclipse.jetty.util.compression.CompressionPool;
import org.eclipse.jetty.util.compression.DeflaterPool;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.resource.ResourceFactory;
import org.jboss.resteasy.plugins.server.servlet.ResteasyBootstrap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.embedded.jetty.JettyServletWebServerFactory;
import org.springframework.boot.web.server.MimeMappings;
import org.springframework.boot.web.servlet.ServletContextInitializer;
import org.springframework.util.Assert;
import org.springframework.web.context.ContextLoader;
import org.springframework.web.filter.CharacterEncodingFilter;
import org.springframework.web.filter.RelativeRedirectFilter;
import org.springframework.web.filter.ShallowEtagHeaderFilter;
import org.springframework.web.servlet.DispatcherServlet;

import java.io.IOException;
import java.net.URI;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static com.xebialabs.deployit.CSPFilter.POLICY_DIRECTIVES_PARAM;

public class DeployJettyServletWebServerFactory extends JettyServletWebServerFactory {

    private static final String SPRING_CTX_CFG_LOCATION_KEY = "contextConfigLocation";
    private static final String CONTEXT_PATH = "deployit";

    @Autowired
    private ServerConfiguration serverConfiguration;

    @Autowired
    private ClientProperties clientProperties;

    @Autowired
    private ServerSideProperties serverSideProperties;

    @Override
    public boolean isRegisterDefaultServlet() {
        return false;
    }

    @Override
    protected void postProcessWebAppContext(WebAppContext contextRoot) {
        setupJetty(contextRoot);
        setupRest(contextRoot);
        setupAfterAuthFilters(contextRoot);
        setSessionHandler(contextRoot);
        setupSessionTimeout(contextRoot);
        setupCSPFilter(contextRoot);
        setupMultipartFilter(contextRoot);

        // setup error handler
        var errorHandler = new CustomErrorHandler(new RootPageMissingErrorHandler(serverConfiguration.getWebWelcomePage()), new LicenseMissingErrorHandler());
        errorHandler.setShowStacks(false);
        errorHandler.setShowMessageInTitle(false);
        contextRoot.setErrorHandler(errorHandler);
    }

    private void setSessionHandler(WebAppContext contextRoot) {
        if (!serverSideProperties.getSession().getActiveUserSessionsEnabled()) {
            SessionHandler sessionHandler = new SessionHandler();
            sessionHandler.setSessionTrackingModes(EnumSet.of(SessionTrackingMode.COOKIE));
            sessionHandler.setSessionCookie("SESSION_XLD");
            sessionHandler.setHttpOnly(true);
            if (!serverConfiguration.isSsl()) {
                sessionHandler.getSessionCookieConfig().setSecure(serverConfiguration.isSecureCookieEnabled());
            }
            sessionHandler.setSameSite(HttpCookie.SameSite.valueOf(SameSiteHelper.getSameSiteValue(
                    SameSiteHelper.getStringValue(
                            clientProperties.sameSiteCookie(), ClientProperties.DEFAULT_SAMESITE_COOKIE())).name()));
            contextRoot.setSessionHandler(sessionHandler);
        }
    }

    private void setupCSPFilter(WebAppContext contextRoot) {
        if (serverSideProperties.getHttp().getCsp().getEnabled()) {
            FilterHolder holder = new FilterHolder(new CSPFilter());
            holder.setAsyncSupported(true);
            holder.setInitParameter(POLICY_DIRECTIVES_PARAM, serverSideProperties.getHttp().getCsp().getPolicyDirectives());
            contextRoot.addFilter(holder, "/*", EnumSet.of(DispatcherType.REQUEST));
        }
    }

    private void setupSessionTimeout(WebAppContext contextRoot) {
        if (!serverSideProperties.getSession().getActiveUserSessionsEnabled()) {
            int sessionTimeoutSeconds = (int) TimeUnit.MINUTES.toSeconds(clientProperties.getSession().getTimeoutMinute());
            contextRoot.getSessionHandler().setMaxInactiveInterval(sessionTimeoutSeconds);
        }
    }

    protected Configuration[] getWebAppContextConfigurations(WebAppContext webAppContext,
                                                             ServletContextInitializer... initializers) {
        List<Configuration> configurations = new ArrayList<>();
        configurations.add(getServletContextInitializerConfiguration(webAppContext, initializers));
        configurations.add(getMimeTypeConfiguration());
        configurations.addAll(getConfigurations());
        return configurations.toArray(new Configuration[0]);
    }

    private Configuration getMimeTypeConfiguration() {
        return new AbstractConfiguration(new AbstractConfiguration.Builder()) {
            @Override
            public void configure(WebAppContext context) {
                MimeTypes.Mutable mimeTypes = context.getMimeTypes();
                for (MimeMappings.Mapping mapping : getMimeMappings()) {
                    mimeTypes.addMimeMapping(mapping.getExtension(), mapping.getMimeType());
                }
            }

        };
    }

    private void setupAfterAuthFilters(WebAppContext contextRoot) {
        FilterHolder maintenanceFilterHolder = contextRoot.addFilter(MaintenanceModeFilter.class, "*",
                EnumSet.of(DispatcherType.REQUEST));

        maintenanceFilterHolder.setInitParameter(
                MaintenanceModeFilter.PARAMETER_MAINTENANCE_FORBIDDEN_REQUEST, String.join(",", clientProperties.maintenance().forbiddenPaths()));
        maintenanceFilterHolder.setInitParameter(
                MaintenanceModeFilter.PARAMETER_CONTEXT_ROOT, serverConfiguration.getWebContextRoot());
    }

    private void setupJetty(WebAppContext contextRoot) {
        contextRoot.addEventListener(new PekkoStreamServletInitializer());
        contextRoot.insertHandler(new CustomInstrumentedHandler(MetricRegistry.getInstance()));
        contextRoot.addFilter(CustomMeasureRequestPathFilter.class, "*", EnumSet.of(DispatcherType.REQUEST))
                .setInitParameter("contentTypes", "application/json|text/html|text/xml|application/xml");

        contextRoot.addFilter(NoOptionsFilter.class, "*", EnumSet.of(DispatcherType.REQUEST));

        if (serverConfiguration.getWebWelcomePage() != null) {
            contextRoot.addFilter(RootPageForwardFilter.class, "/", EnumSet.of(DispatcherType.REQUEST))
                    .setInitParameter(RootPageForwardFilter.FORWARD_TARGET_PARAM, serverConfiguration.getWebWelcomePage());
        }
        contextRoot.addFilter(RequestHeadersEncodedAsParametersFilter.class, "/deployit/reports/*", EnumSet.of(DispatcherType.REQUEST));
        contextRoot.addFilter(RequestHeadersEncodedAsParametersFilter.class, "/deployit/repository/*", EnumSet.of(DispatcherType.REQUEST));
        contextRoot.setClassLoader(Thread.currentThread().getContextClassLoader());

        ServletHolder extensionApiServletHolder = new ServletHolder(FlexPekkoStreamServlet.class);
        extensionApiServletHolder.setDisplayName("ExtensionApiConnectorServlet");
        extensionApiServletHolder.setAsyncSupported(true);
        contextRoot.addEventListener(new PekkoStreamServletInitializer());
        contextRoot.addServlet(extensionApiServletHolder, "/api/*");

        contextRoot.addFilter(new FilterHolder(new AcceptHeaderConditionalFilter(new ShallowEtagHeaderFilter(), MediaType.APPLICATION_JSON)), "/deployit/metadata/*", EnumSet.of(DispatcherType.REQUEST));
        contextRoot.addFilter(NoCacheFilter.class, "/", EnumSet.of(DispatcherType.REQUEST));
        contextRoot.addFilter(CharacterEncodingFilter.class, "/*", EnumSet.of(DispatcherType.REQUEST))
                .setInitParameters(new HashMap<String, String>() {{
                    put("encoding", Charsets.UTF_8.name());
                    put("forceEncoding", "true");
                }});
        contextRoot.addFilter(RelativeRedirectFilter.class, "/*", EnumSet.of(DispatcherType.REQUEST));
        addDeployDefaultServlet(contextRoot);
        if(serverSideProperties.getHttp().getGzip().getEnabled()) {
            enableGzipHandler(contextRoot);
        }
    }

    private void enableGzipHandler(WebAppContext contextRoot) {
        GzipHandler gzipHandler = new GzipHandler();
        GzipProperties gzipSettings = serverSideProperties.getHttp().getGzip();
        gzipHandler.setSyncFlush(true);
        gzipHandler.setIncludedMethods("GET", "POST", "PUT");
        gzipHandler.setDeflaterPool(new DeflaterPool(CompressionPool.DEFAULT_CAPACITY, gzipSettings.getCompression(), true));
        gzipHandler.setMinGzipSize(gzipSettings.minSize() * 1024);
        gzipHandler.setExcludedPaths(gzipSettings.getExcludedPaths().toArray(new String[0]));
        contextRoot.insertHandler(gzipHandler);
    }

    private void setupRest(WebAppContext contextRoot) {
        contextRoot.addEventListener(new ResteasyBootstrap());
        contextRoot.addEventListener(new DeploySpringContextLoaderListener());

        ServletHolder servletHolder = new ServletHolder(DispatcherServlet.class);
        servletHolder.setDisplayName("defaultservlet");
        servletHolder.getInitParameters().put(SPRING_CTX_CFG_LOCATION_KEY, "classpath:spring/deployit-spring-web.xml");
        servletHolder.setInitOrder(1);

        ServletHolder servletCsrfHolder = new ServletHolder(DispatcherServlet.class);
        servletCsrfHolder.setDisplayName("csrfservlet");
        servletCsrfHolder.getInitParameters().put(SPRING_CTX_CFG_LOCATION_KEY, "classpath:spring/csrf-deployit-spring-web.xml");
        servletCsrfHolder.setInitOrder(1);

        contextRoot.getInitParams().put("resteasy.servlet.mapping.prefix", "/" + CONTEXT_PATH);
        contextRoot.getInitParams().put("resteasy.document.expand.entity.references", "false");
        contextRoot.getInitParams().put(ContextLoader.CONTEXT_CLASS_PARAM, XLWebApplicationContext.class.getName());

        contextRoot.addServlet(servletHolder, "/" + CONTEXT_PATH + "/*");
        contextRoot.addServlet(servletCsrfHolder, "/xldeploy/*");
    }

    private void addDeployDefaultServlet(WebAppContext context) {
        Assert.notNull(context, "Context must not be null");
        ServletHolder holder = new ServletHolder();
        holder.setName("default");
        holder.setServlet(new DefaultServlet());
        holder.setInitParameter("dirAllowed", "false");
        URI path = null;
        Resource baseResource = null;
        try {
            ResourceFactory resourceFactory = context.getResourceFactory();
            path = Path.of(".").toRealPath(LinkOption.NOFOLLOW_LINKS).toUri();
            baseResource = resourceFactory.newResource(path);
        } catch (IOException e) {
            throw new RuntimeException("Exception occurred while setting base resource", e);
        }
        List<Resource> resources = new ArrayList<>();
        resources.add(baseResource);
        resources.add(new XLDWebResource());
        Resource resource = ResourceFactory.combine(resources);
        holder.setInitOrder(1);
        context.getServletHandler().addServletWithMapping(holder, "/");
        context.getServletHandler().getServletMapping("/").setFromDefaultDescriptor(true);
        context.getServletHandler().setDecodeAmbiguousURIs(true);
        context.setBaseResource(resource);
    }

    private void setupMultipartFilter(WebAppContext context) {
        FilterHolder multipartFilter = new FilterHolder(MultipartFilter.class);
        multipartFilter.setName("multipartFilter");
        context.addFilter(multipartFilter, "/*", EnumSet.of(DispatcherType.REQUEST));
    }
}
