package com.xebialabs.deployit.server.jetty.metrics;

import com.tqdev.metrics.core.MetricRegistry;
import jakarta.servlet.*;
import jakarta.servlet.http.HttpServletRequest;

import java.io.IOException;

public class CustomMeasureRequestPathFilter implements Filter {
    private final MetricRegistry registry;

    /**
     * The content types for which the path is grouped, e.g: "json|xml|html|csv"
     */
    private String contentTypes;

    public CustomMeasureRequestPathFilter() {
        this(MetricRegistry.getInstance(), null);
    }

    public CustomMeasureRequestPathFilter(MetricRegistry registry, String contentTypes) {
        this.registry = registry;
        this.contentTypes = contentTypes;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        if (this.contentTypes == null) {
            this.contentTypes = filterConfig.getInitParameter("contentTypes");
        }
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        if (!registry.isEnabled()) {
            chain.doFilter(request, response);
            return;
        }

        if (!(request instanceof HttpServletRequest)) {
            chain.doFilter(request, response);
            return;
        }

        HttpServletRequest httpRequest = (HttpServletRequest) request;
        String requestURI = httpRequest.getRequestURI();

        final long startTime = registry.getNanos();
        try {
            chain.doFilter(request, response);
        } finally {
            final long duration = registry.getNanos() - startTime;
            registerMeasurement(requestURI, response.getContentType(), duration);
        }
    }

    private void registerMeasurement(String requestURI, String contentType, long duration) {
        final String pathGroup = getPathGroup(requestURI, contentType);
        if (pathGroup != null) {
            registry.increment("http.Path.Invocations", pathGroup);
            registry.add("http.Path.Durations", pathGroup, duration);
        }
    }

    /**
     * Get a grouping identifier for metrics based on path.
     *
     * @param requestURI
     *            the request URI
     * @param contentType
     *            the content type
     * @return the string
     */
    private String getPathGroup(String requestURI, String contentType) {
        if (requestURI == null || contentType == null || !contentType.matches(".*(" + contentTypes + ").*")) {
            return "(other)";
        }
        String parts[] = requestURI.split("/");
        for (int i = 0; i < parts.length; i++) {
            if (parts[i].matches("[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}")) {
                parts[i] = "(uuid)";
            } else if (parts[i].matches("[a-f0-9]{128}")) {
                parts[i] = "(sha512)";
            } else if (parts[i].matches("[a-f0-9]{64}")) {
                parts[i] = "(sha256)";
            } else if (parts[i].matches("[a-f0-9]{40}")) {
                parts[i] = "(sha1)";
            } else if (parts[i].matches("[a-f0-9]{32}")) {
                parts[i] = "(md5)";
            } else if (parts[i].matches("[^a-zA-Z]+") && parts[i].matches(".*[0-9].*")) {
                parts[i] = "(number)";
            }
        }
        String path = String.join("/", parts);
        if (path.isEmpty()) {
            path = "/";
        }
        return path;
    }

    @Override
    public void destroy() {
        // nothing yet
    }
}
