package com.xebialabs.deployit

import ai.digital.deploy.core.common.XldServerPaths
import com.typesafe.config.ConfigFactory
import com.xebialabs.deployit.configuration.system.SystemValues
import com.xebialabs.deployit.core.config.{SqlConfiguration, TrueZipShutdownConfiguration}
import com.xebialabs.deployit.core.config.session.SpringJdbcSessionConfig
import com.xebialabs.deployit.resolver.XLPluginAwareResourcePatternResolver
import com.xebialabs.deployit.upgrade.{CentralConfigVersionFile, PlatformUpgrade, UpgradeStrategy}
import com.xebialabs.license.LicenseRegistrationServlet
import com.xebialabs.license.service.{LicenseInstallServiceImpl, LicenseService}
import com.xebialabs.xldeploy.jms.adapter.DefaultQueueJmsBean
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService
import grizzled.slf4j.Logging
import jakarta.servlet.{DispatcherType, Filter}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.beans.factory.config.AutowireCapableBeanFactory
import org.springframework.boot.actuate.autoconfigure.security.servlet.ManagementWebSecurityAutoConfiguration
import org.springframework.boot.autoconfigure.EnableAutoConfiguration
import org.springframework.boot.autoconfigure.flyway.FlywayAutoConfiguration
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration
import org.springframework.boot.autoconfigure.security.oauth2.client.reactive.ReactiveOAuth2ClientAutoConfiguration
import org.springframework.boot.autoconfigure.security.servlet.SecurityAutoConfiguration
import org.springframework.boot.autoconfigure.web.servlet.{MultipartAutoConfiguration, WebMvcAutoConfiguration}
import org.springframework.boot.web.servlet.{FilterRegistrationBean, ServletRegistrationBean}
import org.springframework.boot.{CommandLineRunner, SpringApplication}
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation._
import org.springframework.security.web.context.AbstractSecurityWebApplicationInitializer

import java.io.File
import java.nio.file.Files
import java.util.{HashMap => JHashMap}
import scala.jdk.CollectionConverters._

object ApplicationBootstrapper {

  def main(args: Array[String]): Unit = {
    SystemValues.apply()
    ApplicationBooter.boot(args)

    val app = new SpringApplication(classOf[DeployitBootstrapper])
    app.setResourceLoader(new XLPluginAwareResourcePatternResolver())
    app.setBanner(new XLDeployBanner())
    app.setAllowBeanDefinitionOverriding(true)
    app.run(args: _*)
  }
}

@Configuration
@EnableAutoConfiguration(
  exclude = Array(
    classOf[DataSourceAutoConfiguration],
    classOf[FlywayAutoConfiguration],
    classOf[MultipartAutoConfiguration],
    classOf[ReactiveOAuth2ClientAutoConfiguration],
    classOf[SecurityAutoConfiguration],
    classOf[ManagementWebSecurityAutoConfiguration],
    classOf[WebMvcAutoConfiguration],
  )
)
@Import(
  Array(
    classOf[DefaultQueueJmsBean],
    classOf[SqlConfiguration],
    classOf[SpringJdbcSessionConfig],
    classOf[TrueZipShutdownConfiguration]
  )
)
@ImportResource(
  Array(
    "classpath:spring/deployit-context.xml",
    "classpath*:deployit-security.xml",
    "classpath*:hotfix-context.xml"
  )
)
@EnableAspectJAutoProxy
class ApplicationBootstrapper extends CommandLineRunner with Logging {

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var beanFactory: AutowireCapableBeanFactory = _

  @Autowired
  var upgradeStrategy: UpgradeStrategy = _

  @Autowired
  var serverLaunchOptions: ServerLaunchOptions = _

  @Autowired
  var repositoryVersionService: RepositoryVersionService = _

  @Autowired
  var applicationContext: ApplicationContext = _

  @Bean
  def licenseInstallService: LicenseInstallServiceImpl =
    new LicenseInstallServiceImpl("conf/deployit-license.lic", licenseService)

  @Bean
  def licenseRegistrationServlet: ServletRegistrationBean[LicenseRegistrationServlet] = {
    val srb = new ServletRegistrationBean[LicenseRegistrationServlet]
    val servlet = new LicenseRegistrationServlet()
    beanFactory.autowireBean(servlet)
    srb.setServlet(servlet)
    srb.setInitParameters(new JHashMap[String, String]() {
      put(LicenseRegistrationServlet.PRODUCT_NAME, "Digital.ai Deploy")
      put(LicenseRegistrationServlet.PRODUCT_LOGO_URL, "images/xl_deploy_logo_green.svg")
      put(LicenseRegistrationServlet.ACTIVATION_URL, "https://xebialabs.com/products/xl-deploy/trial/activation")
    })
    srb.setUrlMappings(List("/productregistration/*").asJava)
    srb
  }

  @Bean
  def registerAPIFilters(@Qualifier(AbstractSecurityWebApplicationInitializer.DEFAULT_FILTER_NAME) securityFilter: Filter) = {
    val registrationBean = new FilterRegistrationBean(securityFilter)
    registrationBean.setIgnoreRegistrationFailure(true)
    registrationBean.setName(AbstractSecurityWebApplicationInitializer.DEFAULT_FILTER_NAME)
    registrationBean.setUrlPatterns(
      List(
        "/",
        "/login",
        "/login/*",
        "/logout",
        "/oauth2/authorization/xl-deploy",
        "/deployit/*",
        "/api/*",
        "/xldeploy/*",
        "/*",
        "/centralConfiguration/*"
      ).asJava
    )
    registrationBean.setDispatcherTypes(java.util.EnumSet.of(DispatcherType.REQUEST))
    registrationBean
  }

  @Bean
  @DependsOn(Array("repositoryInitializer"))
  def upgrader: PlatformUpgrade = {
    val forceUpgrades = serverLaunchOptions.isForceUpgrades
    val centralConfigVersionFile = new CentralConfigVersionFile(XldServerPaths.DEFAULT_CENTRAL_CONFIG_VERSION_FILE,
      XldServerPaths.DEFAULT_CONFIGURATION_FILE, XldServerPaths.CONFIG_FOLDER)
    val upgrader = new PlatformUpgrade(
      upgradeStrategy, forceUpgrades, repositoryVersionService, centralConfigVersionFile, postConfigUpgrade)
    upgrader.setApplicationContext(applicationContext)
    upgrader.autoUpgrade()
    upgrader
  }

  private def postConfigUpgrade(): Unit = {
    val xlDeployConfFile = new File(XldServerPaths.CONFIG_FOLDER, "xl-deploy.conf")
    if (xlDeployConfFile.exists()) {
      val config = ConfigFactory.parseFile(xlDeployConfFile)
      if (config.isEmpty || config.entrySet().isEmpty) {
        logger.info(s"Deleting xl-deploy.conf from ${xlDeployConfFile.getAbsolutePath} " +
          s" since it does not have any configuration $config")
        Files.delete(xlDeployConfFile.toPath)
      } else {
        logger.info(s"Not deleting xl-deploy.conf since it contains configuration - $config")
      }
    }
  }

  override def run(args: String*): Unit = {
    ApplicationBooter.setUpTrustStorePassword()
  }
}
