/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.setup;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.MalformedURLException;

import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.core.io.UrlResource;

import com.xebialabs.deployit.DeployitOptions;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.jcr.JackrabbitRepositoryFactoryBean;

public class Setup {
	private static final int DEFAULT_HTTP_PORT = 4516;
	private static final int DEFAULT_HTTPS_PORT = 4517;
	private static final int DEFAULT_MAX_THREADS = 24;
	private static final int DEFAULT_MIN_THREADS = 3;
	private static final String FILE_URL_PREFIX = "file://";
	private static final String DEFAULT_JCR_REPOSITORY_PATH = "repository";
	private static final String DEFAULT_IMPORTABLE_PACKAGES_PATH = "importablePackages";
	private static final String EXITSETUP_KEYWORD = "exitsetup";

	public static void setup(DeployitOptions deployitOptions) {
		Setup setup = new Setup();
		if (deployitOptions.isReinitialize()) {
			setup.reinitialize();
		} else {
			setup.run();
		}
	}

	private void reinitialize() {
		SetupContext context = new SetupContext();
		useDefaultValues(context);
		context.save();
		initializeJcrRepository(context);
		deleteTaskRecoveryFile();
	}

	private void run() {
		System.out.println("\nWelcome to the Deployit setup.");
		System.out.println("You can always exit by typing '" + EXITSETUP_KEYWORD + "'.");
		System.out
		        .println("To re-run this setup and make changes to the Deployit server configuration you can run server.cmd -setup on Windows or server.sh -setup on Unix.");

		SetupContext context = new SetupContext();

		boolean editingExistingConfiguration = false;
		if (context.exists()) {
			if (askDoYouWantToEditTheExistingConfiguration(context)) {
				context.load();
				editingExistingConfiguration = true;
			}
		}

		if (askDoYouWantToUseTheSimpleSetup()) {
			if (!editingExistingConfiguration) {
				useDefaultValues(context);
			} else {
				upgradeConfigurationWithDefaultValues(context);
			}
			askToInitializeJcr(context);
		} else {
			if (askToEnableSsl(context)) {
				if (askToGenerateKeys(context)) {
					setDefaultKeyStoreSettings(context);
				} else {
					askKeyStoreSettings(context);
				}
			}

			askHttpPortForJetty(context);
			askMinumAmountOfThreads(context);
			askMaximumAmountOfThreads(context);

			askJcrRepositoryPath(context);
			askToInitializeJcr(context);

			askImprotablePackagesPath(context);
		}

		if (reviewAndConfirm(context)) {
			System.out.println("Saving to " + context.getLocation());
			context.save();
			System.out.println("Configuration saved.");
			if (context.isGenerateKeyStore()) {
				generateKeyStore(context);
			}

			if (context.isInitializeJcrRepository()) {
				initializeJcrRepository(context);
			}

			System.out.println("You can now start your Deployit server by executing the command server.cmd on Windows or server.sh on Unix.");
			System.out.println("Note: If your Deployit server is running please restart it.");
			System.out.println("Finished setup.");
		} else {
			System.out.println("Aborting setup.");
		}
	}

	private void upgradeConfigurationWithDefaultValues(SetupContext context) {
		if (context.getMinThreads() == 0) {
			context.setMinThreads(DEFAULT_MIN_THREADS);
		}
		if (context.getMaxThreads() == 0) {
			context.setMaxThreads(DEFAULT_MAX_THREADS);
		}
		if (context.getImportablePackagesPath() == null) {
			context.setImportablePackagesPath(DEFAULT_IMPORTABLE_PACKAGES_PATH);
		}
	}

	private void askMinumAmountOfThreads(SetupContext context) {
		printEmptyLine();
		System.out.println("Enter the minimum number of threads the HTTP server should use (recommended: 3 per client, so 3 for single user usage)");
		int minThreads = DEFAULT_MIN_THREADS;
		int suppliedMinThreads = (context.getMinThreads() == 0) ? minThreads : context.getMinThreads();
		context.setMinThreads(getValidIntegerResponse(suppliedMinThreads));
	}

	private void askMaximumAmountOfThreads(SetupContext context) {
		printEmptyLine();
		System.out.println("Enter the maximum number of threads the HTTP server should use (recommended :3 per client, so 24 for 8 concurrent users)");
		int maxThreads = DEFAULT_MAX_THREADS;
		int suppliedMaxThreads = (context.getMaxThreads() == 0) ? maxThreads : context.getMaxThreads();
		context.setMaxThreads(getValidIntegerResponse(suppliedMaxThreads));
	}

	private void useDefaultValues(SetupContext context) {
		context.setJcrRepositoryPath(DEFAULT_JCR_REPOSITORY_PATH);
		context.setHttpPort(DEFAULT_HTTP_PORT);
		context.setImportablePackagesPath(DEFAULT_IMPORTABLE_PACKAGES_PATH);
		context.setMinThreads(DEFAULT_MIN_THREADS);
		context.setMaxThreads(DEFAULT_MAX_THREADS);
	}

	private boolean askDoYouWantToUseTheSimpleSetup() {
		printEmptyLine();
		System.out.println("Do you want to use the simple setup?");
		System.out.println("Default values are used for all properties. To make changes to the default properties, please answer no.");
		System.out.println("Options are yes or no.");
		return getBooleanResponse(true);
	}

	private boolean askToEnableSsl(SetupContext context) {
		printEmptyLine();
		System.out.println("Would you like to enable SSL?");
		System.out.println("Options are yes or no.");
		boolean suppliedSsl = context.isNewConfiguration() || context.isSsl();
		context.setSsl(getBooleanResponse(suppliedSsl));
		return context.isSsl();
	}

	private boolean askToGenerateKeys(SetupContext context) {
		printEmptyLine();
		System.out.println("Would you like Deployit to generate a keystore with a self-signed certificate for you?");
		System.out.println("N.B.: Self-signed certificates do not work correctly with some versions of the Flash Player and some browsers!");
		System.out.println("Options are yes or no.");
		boolean suppliedGenerateKeys = context.isNewConfiguration() || context.isGenerateKeyStore();
		context.setGenerateKeyStore(getBooleanResponse(suppliedGenerateKeys));
		return context.isGenerateKeyStore();
	}

	private void setDefaultKeyStoreSettings(SetupContext context) {
		context.setKeyStorePath("conf/keystore.jks");
		context.setKeyStorePassword("storesecret");
		context.setKeyStoreKeyPassword("keysecret");
	}

	private void askKeyStoreSettings(SetupContext context) {
		printEmptyLine();
		System.out.println("What is the path to the keystore?");
		String suppliedKeyStorePath = context.isNewConfiguration() ? "" : context.getKeyStorePath();
		context.setKeyStorePath(getStringResponse(suppliedKeyStorePath));

		printEmptyLine();
		System.out.println("What is the password to the keystore?");
		String suppliedKeyStorePassword = context.isNewConfiguration() ? "" : context.getKeyStorePassword();
		context.setKeyStorePassword(getStringResponse(suppliedKeyStorePassword));

		printEmptyLine();
		System.out.println("What is the password to the key in the keystore?");
		String suppliedKeyStoreKeyPassword = context.isNewConfiguration() ? "" : context.getKeyStoreKeyPassword();
		context.setKeyStoreKeyPassword(getStringResponse(suppliedKeyStoreKeyPassword));
	}

	private void askHttpPortForJetty(SetupContext context) {
		printEmptyLine();
		System.out.println("What http port number would you like the server to listen on?");
		int defaultPort = context.isSsl() ? DEFAULT_HTTPS_PORT : DEFAULT_HTTP_PORT;
		int suppliedPort = (context.getHttpPort() == 0) ? defaultPort : context.getHttpPort();
		context.setHttpPort(getValidIntegerResponse(suppliedPort));
	}

	private void askImprotablePackagesPath(SetupContext context) {
		printEmptyLine();
		System.out.println("Where would you like Deployit to import packages from?");
		String suppliedApplicationImportPath = (context.getImportablePackagesPath() == null) ? DEFAULT_IMPORTABLE_PACKAGES_PATH : context
		        .getImportablePackagesPath();
		context.setImportablePackagesPath(getValidNotEmptyStringResponse(suppliedApplicationImportPath));
	}

	private boolean askDoYouWantToEditTheExistingConfiguration(SetupContext context) {
		boolean useOldConfig;
		printEmptyLine();
		System.out.println("An existing configuration was found. Do you want to edit it?");
		System.out.println("Options are yes or no. (selecting no will create an empty configuration)");
		useOldConfig = getBooleanResponse(true);
		if (useOldConfig)
			System.out.println("Editing the exisiting configuration.");
		else
			System.out.println("Starting configuration from scratch.");
		return useOldConfig;
	}

	private void askJcrRepositoryPath(SetupContext context) {
		printEmptyLine();
		System.out.println("Where would you like Deployit to store the JCR repository?");
		String path = (context.getJcrRepositoryPath() == null) ? DEFAULT_JCR_REPOSITORY_PATH : context.getJcrRepositoryPath();
		String userEnteredRepoPath = getValidNotEmptyStringResponse(path);
		context.setJcrRepositoryPath(convertAbsolutePathToFileUrlIfNeeded(userEnteredRepoPath));
	}

	private String convertAbsolutePathToFileUrlIfNeeded(String path) {
		if (path.startsWith("/")) {
			return FILE_URL_PREFIX + path;
		}
		return path;
	}

	private boolean askToInitializeJcr(SetupContext context) {
		printEmptyLine();
		System.out.println("Do you want Deployit to initialize the JCR repository?");
		System.out.println("Options are yes or no.");
		if (context.isNewConfiguration()) {
			context.setInitializeJcrRepository(getBooleanResponse(true));
		} else {
			context.setInitializeJcrRepository(getBooleanResponse(false));
		}
		if (context.isInitializeJcrRepository())
			System.out.println("Deployit will initialize the JCR repository.");
		else
			System.out.println("Deployit will NOT initialize the JCR repository.");
		return context.isInitializeJcrRepository();
	}

	private boolean reviewAndConfirm(SetupContext context) {
		printEmptyLine();
		System.out.println("Do you agree with the following settings for Deployit and would you like to save them?");
		System.out.println("Changes will be saved in deployit.conf");
		System.out.println("\tSSL will be " + (context.isSsl() ? "enabled" : "disabled"));
		if (context.isSsl()) {
			if (context.isGenerateKeyStore()) {
				System.out.println("\tKeystore will be generated");
			} else {
				System.out.println("\tKeystore path is " + context.getKeyStorePath());
				System.out.println("\tKeystore password is " + context.getKeyStorePassword());
				System.out.println("\tKeystore key password is " + context.getKeyStoreKeyPassword());
			}
		}
		System.out.println("\tHTTP port is " + context.getHttpPort());
		System.out.println("\tHTTP server will use a minimum of " + context.getMinThreads() + " and a maximum of " + context.getMaxThreads() + " threads");
		try {
			System.out.println("\tJCR repository home is at " + resolveRepositoryHome(context).getFile().getAbsolutePath());
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		if (context.isInitializeJcrRepository()) {
			System.out.println("\tJCR repository will be initialized.");
			System.out.println("\tTask recovery file will deleted");
		}
		System.out.println("\tApplication import location is " + context.getImportablePackagesPath());

		return getBooleanResponse(true);
	}

	private void generateKeyStore(SetupContext context) {
		String keyAlgorithm = "RSA";
		String keyStoreKeyAlias = "jetty";
		String dname = "CN=localhost,O=Deployit,C=NL";

		try {
			File keyStoreFile = new File(context.getKeyStorePath());
			if (keyStoreFile.exists()) {
				System.out.println("Existing keystore " + context.getKeyStorePath() + " deleted.");
				keyStoreFile.delete();
			}
			System.out.println("Generating keystore...");
			String[] keytoolArgs = new String[] { "keytool", "-genkey", "-keyalg", keyAlgorithm, "-keystore", context.getKeyStorePath(), "-storepass",
			        context.getKeyStorePassword(), "-alias", keyStoreKeyAlias, "-keypass", context.getKeyStoreKeyPassword(), "-validity", "366", "-dname",
			        dname };
			Process p = Runtime.getRuntime().exec(keytoolArgs);

			Thread outputThread = new Thread(new InputStreamToOutputStream(p.getInputStream(), System.out));
			outputThread.start();
			Thread errorThread = new Thread(new InputStreamToOutputStream(p.getErrorStream(), System.err));
			errorThread.start();

			int exitValue;
			try {
				exitValue = p.waitFor();
			} catch (InterruptedException exc) {
				throw new IOException(exc.toString());
			}
			if (exitValue != 0) {
				throw new IOException("keytool exited with status code " + exitValue);
			}
			System.out.println("Keystore generated.");
		} catch (IOException exc) {
			System.err.println("WARNING: Could not generate keystore " + context.getKeyStorePath() + ": " + exc.toString());
		}
	}

	private static class InputStreamToOutputStream implements Runnable {
		private InputStream in;
		private OutputStream out;

		public InputStreamToOutputStream(InputStream in, OutputStream out) {
			this.in = in;
			this.out = out;
		}

		public void run() {
			try {
				ByteStreams.copy(in, out);
			} catch (IOException ignore) {
			}
		}
	}

	private void initializeJcrRepository(SetupContext context) {
		Resource homeDirResource = resolveRepositoryHome(context);
		cleanUpAndReinitializeRepositoryHome(homeDirResource);

		final JackrabbitRepositoryFactoryBean repositoryFactoryBean = new JackrabbitRepositoryFactoryBean();
		repositoryFactoryBean.setConfiguration(new ClassPathResource("jackrabbit-repository.xml"));
		repositoryFactoryBean.setHomeDir(homeDirResource);
		repositoryFactoryBean.setCreateHomeDirIfNotExists(true);

		try {
			repositoryFactoryBean.afterPropertiesSet();
		} catch (Exception exc) {
			throw new RuntimeException("Cannot initialize JCR repository", exc);
		}

		repositoryFactoryBean.configureJcrRepositoryForDeployit();
		repositoryFactoryBean.destroy();

		System.out.println("JCR repository initialized.");
		deleteTaskRecoveryFile();
	}

	private Resource resolveRepositoryHome(SetupContext context) {
		Resource homeDirResource;
		if (context.getJcrRepositoryPath().startsWith(FILE_URL_PREFIX)) {
			try {
				homeDirResource = new UrlResource(context.getJcrRepositoryPath());
			} catch (MalformedURLException e) {
				throw new RuntimeException("JCR Repository path is a malformed url. " + context.getJcrRepositoryPath());
			}
		} else {
			homeDirResource = new FileSystemResource(context.getJcrRepositoryPath());
		}
		return homeDirResource;
	}

	private void cleanUpAndReinitializeRepositoryHome(Resource repoHomeResource) {
		File repositoryHome = null;
		try {
			repositoryHome = repoHomeResource.getFile();
		} catch (IOException e) {
			throw new RuntimeIOException("Problem resolving absolute path to JCR repository at " + repoHomeResource.getFilename(), e);
		}

		if (repositoryHome.exists()) {
			System.out.println("Deleting JCR repository at " + repositoryHome);
			try {
				Files.deleteRecursively(repositoryHome);
			} catch (IOException exc) {
				throw new RuntimeIOException("Cannot delete JCR repository at " + repositoryHome, exc);
			}
		}

		System.out.println("Creating JCR repository at " + repositoryHome);
		final boolean created = repositoryHome.mkdirs();
		if (!created)
			throw new RuntimeIOException("Unknown error occured while creating JCR repository at " + repoHomeResource);
	}

	private void deleteTaskRecoveryFile() {
		File taskRecoveryFile = new File("recovery.dat");
		if (taskRecoveryFile.exists()) {
			if (taskRecoveryFile.delete()) {
				System.out.println("Task recovery file deleted");
			} else {
				System.err.println("Couldn't delete the task recovery file");
			}
		}
	}

	private void printEmptyLine() {
		System.out.println("");
	}

	private String read() {
		BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
		try {
			final String line = stdin.readLine();
			if (line != null) {
				String value = line.trim();
				if (EXITSETUP_KEYWORD.equals(value)) {
					System.out.println("Exiting setup.");
					System.exit(1);
				}
				return value;
			} else
				return line;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	private boolean getBooleanResponse(boolean defaultValue) {
		String prompt = determinePrompt((defaultValue) ? "yes" : "no");

		while (true) {
			System.out.print(prompt);
			String value = read();
			if ("".equals(value))
				return defaultValue;
			if ("yes".equalsIgnoreCase(value))
				return true;
			if ("no".equalsIgnoreCase(value))
				return false;
			System.out.println("Illegal value specified, type yes or no.");
		}
	}

	private String getValidNotEmptyStringResponse(String defaultValue) {
		String prompt = determinePrompt(defaultValue);

		while (true) {
			System.out.print(prompt);
			String value = read();
			if ("".equals(value) && defaultValue != null)
				value = defaultValue;

			if (!"".equals(value) && value != null)
				return value;
			System.out.println("Value cannot be empty.");
		}
	}

	private String getStringResponse(String defaultValue) {
		String prompt = determinePrompt(defaultValue);

		while (true) {
			System.out.print(prompt);
			String value = read();
			if ("".equals(value))
				value = defaultValue;
			if (value != null)
				return value;
		}
	}

	private int getValidIntegerResponse(int defaultValueAsAnInt) {
		String defaultValue = Integer.toString(defaultValueAsAnInt);
		String prompt = determinePrompt(defaultValue);

		while (true) {
			System.out.print(prompt);
			String value = read();
			if ("".equals(value))
				value = defaultValue;
			try {
				int returnedValue = Integer.parseInt(value);
				if (returnedValue > 0) {
					return returnedValue;
				} else {
					System.out.println("Illegal value specified. Number must be larger then 0.");
				}
			} catch (NumberFormatException nfe) {
				System.out.println("Illegal value specified. Must be a number.");
			}
		}
	}

	private String determinePrompt(String defaultValue) {
		String prompt;
		if (defaultValue != null)
			prompt = "[" + defaultValue + "]: ";
		else
			prompt = ": ";
		return prompt;
	}

}
