package com.xebialabs.deployit.test.support.onthefly;

import static com.xebialabs.deployit.test.support.utils.VelocityUtils.evaluateTemplate;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.Servlet;

import org.apache.velocity.VelocityContext;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;

/**
 * Constructs a WAR for testing, accepting multiple {@link Resource resources} and test {@link Servlet servlets}
 * as entries. Test servlets are automatically mapped to a path that is the servlet class' simple name,
 * so e.g. a {@code com.acme.package.TestJeeResourceServlet} will be bound to {@code http://myhost:myport/context-root/TestJeeResourceServlet}.
 * The actual WAR file is finally written to a temporary directory. Example:
 * 
 * <pre>
 * WarOnTheFly warOnTheFly = new WarOnTheFly("myWar");
 * warOnTheFly.addFile("foo", someFile);
 * warOnTheFly.addFile("bar", someResource);
 * ...
 * warOnTheFly.addServlet(FooServlet.class); // will be mapped to http://myhost:myport/context-root/FooServlet
 * warOnTheFly.addServlet(BarServlet.class);
 * ...
 * File war = warOnTheFly.writeToTemporaryFile(); // something like myWarXX.war
 * </pre>
 * 
 * See also &quot;<a href="http://blog.xebia.com/2009/12/14/middleware-integration-testing-with-junit-maven-and-vmware-part-2/">
 * Middleware integration testing with JUnit, Maven and VMware: part 2</a>&quot;.
 * 
 * @see JarOnTheFly
 */
public class WarOnTheFly extends JarOnTheFly {

	private String name;

	private Map<String, String> servlets;

	public WarOnTheFly(String name) {
		this.name = name;
		this.servlets = new HashMap<String, String>();
	}

	public void addServlet(Class<? extends Servlet> servletClass) {
		String servletName = servletClass.getSimpleName();
		String servletClassFilename = servletClass.getName().replace('.', '/') + ".class";
		addFile("WEB-INF/classes/" + servletClassFilename, new ClassPathResource(servletClassFilename));
		servlets.put(servletName, servletClass.getName());
	}

	public File writeToTemporaryFile() throws IOException {
		addWebXml();
		return writeToTemporaryFile(name, ".war");
	}

	public void addWebXml() {
		VelocityContext context = new VelocityContext();
		context.put("name", name);
		context.put("servlets", servlets);
		String webxml = evaluateTemplate(context, "com/xebialabs/deployit/test/support/onthefly/web.xml.vm");
		addFile("WEB-INF/web.xml", new ByteArrayResource(webxml.getBytes()));
	}

	public String getName() {
		return name;
	}

}
