package com.xebialabs.deployit.plugin.tomcat;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Iterator;
import java.util.List;

import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatDataSource;

public class TomcatContextFileUtils {

	private HostFile contextFile;

	public TomcatContextFileUtils(HostFile contextFile) {
		this.contextFile = contextFile;
	}

	public boolean exists() {
		return contextFile.exists();
	}

	public boolean delete() {
		return contextFile.delete();
	}

	public void addOrUpdateContextPath(String docBase) {
		Document doc = getDocument();
		findAndRemoveExistingDocBase(doc);
		Element contextElement = getOrCreateContext(doc);
		contextElement.addAttribute("docBase", docBase);
		saveDocumentInContextFile(doc);
	}

	public void addOrUpdateDataSource(TomcatDataSource ds) {
		Document doc = getDocument();
		findAndRemoveExistingDatasource(ds, doc);
		Element context = getOrCreateContext(doc);
		Element newResourceDSElement = context.addElement("Resource");
		newResourceDSElement.addAttribute("name", ds.getJndiName());
		newResourceDSElement.addAttribute("url", ds.getConnectionUrl());
		newResourceDSElement.addAttribute("auth", "Container");
		newResourceDSElement.addAttribute("type", "javax.sql.DataSource");
		newResourceDSElement.addAttribute("username", ds.getUsername());
		newResourceDSElement.addAttribute("password", ds.getPassword());
		newResourceDSElement.addAttribute("driverClassName", ds.getDriverClass());
		if (ds.getMaxActive() > 0) {
			newResourceDSElement.addAttribute("maxActive", Integer.toString(ds.getMaxActive()));
		}
		if (ds.getMaxIdle() > 0) {
			newResourceDSElement.addAttribute("maxIdle", Long.toString(ds.getMaxIdle()));
		}

		saveDocumentInContextFile(doc);
	}

	public boolean deleteDataSource(TomcatDataSource ds) {
		Document doc = getDocument();
		findAndRemoveExistingDatasource(ds, doc);
		saveDocumentInContextFile(doc);
		return true;
	}

	private Document getDocument() {
		Document doc;
		if (contextFile.exists()) {
			SAXReader reader = new SAXReader();
			try {
				doc = reader.read(contextFile.get());
			} catch (RuntimeIOException e) {
				throw new RuntimeIOException("Unable to parse existing context xml file " + contextFile.getName(), e);
			} catch (DocumentException e) {
				throw new RuntimeIOException("Unable to parse existing context xml file " + contextFile.getName(), e);
			}
		} else {
			doc = DocumentHelper.createDocument();
		}
		return doc;
	}

	@SuppressWarnings("unchecked")
	private void findAndRemoveExistingDatasource(TomcatDataSource ds, Document doc) {
		List resourceTypeList = doc.selectNodes("//Context/Resource/@type");
		for (Iterator iter = resourceTypeList.iterator(); iter.hasNext();) {
			Attribute type = (Attribute) iter.next();
			String value = type.getValue();
			if (value.equalsIgnoreCase("javax.sql.DataSource")) {
				Element resource = type.getParent();
				Attribute name = (Attribute) resource.selectSingleNode("@name");
				String nameValue = name.getValue();
				if (nameValue.equals(ds.getJndiName())) {
					Element parent = resource.getParent();
					parent.remove(resource);
				}
			}
		}
	}

	private void findAndRemoveExistingDocBase(Document doc) {
		Attribute docBaseAttr = (Attribute) doc.selectSingleNode("//Context/@docBase");
		if (docBaseAttr != null) {
			docBaseAttr.getParent().remove(docBaseAttr);
		}
	}

	private void saveDocumentInContextFile(Document doc) {
		StringWriter stringWriter = new StringWriter();
		XMLWriter xmlWriter = new XMLWriter(stringWriter);
		try {
			xmlWriter.write(doc);
			xmlWriter.close();
		} catch (IOException e) {
			throw new RuntimeIOException("Unable to add or update Tomcat context xml file " + contextFile.getName(), e);
		}
		HostFileUtils.putStringToHostFile(stringWriter.toString(), contextFile);
	}

	private Element getOrCreateContext(Document doc) {
		Element contextElement;
		if (doc.selectSingleNode("//Context") == null) {
			contextElement = doc.addElement("Context");
		} else {
			contextElement = (Element) doc.selectSingleNode("Context");
		}
		return contextElement;
	}

}
