/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.tomcat;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostFileUtils;
import com.xebialabs.deployit.plugin.tomcat.ci.ResourceLink;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatActiveMQConnectionFactory;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatDataSource;
import com.xebialabs.deployit.plugin.tomcat.ci.TomcatResource;
import com.xebialabs.deployit.plugin.tomcat.utils.CiToXML;
import org.dom4j.*;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Iterator;
import java.util.List;

import static java.lang.String.format;

public class TomcatContextFileUtils {

	private HostFile contextFile;


	public TomcatContextFileUtils(HostFile contextFile) {
		this.contextFile = contextFile;
	}

	public boolean exists() {
		return contextFile.exists();
	}

	public boolean delete() {
		return contextFile.delete();
	}

	public void addOrUpdateContextPath(String docBase) {
		Document doc = getDocument();
		findAndRemoveExistingDocBase(doc);
		Element contextElement = getOrCreateContext(doc);
		contextElement.addAttribute("docBase", docBase);
		saveDocumentInContextFile(doc);
	}

	public void addOrUpdateDataSource(TomcatDataSource ds) {
		Document doc = getDocument();
		findAndRemoveExistingDatasource(ds, doc);
		Element context = getOrCreateContext(doc);

		Element newResourceDSElement = context.addElement("Resource");

		final CiToXML toXml = new CiToXML(ds, newResourceDSElement);
		toXml.addTranslation("jndiName", "name");
		toXml.addTranslation("connectionUrl", "url");
		toXml.addTranslation("driverClass", "driverClassName");

		toXml.addAttribute("auth", "Container");
		toXml.addAttribute("type", TomcatDataSource.TYPE);

		toXml.process();

		saveDocumentInContextFile(doc);
	}

	public void addOrUpdateTomcatActiveMQConnectionFactory(TomcatActiveMQConnectionFactory amqcf) {
		Document doc = getDocument();
		findAndRemoveExistingAMQCF(amqcf, doc);
		Element context = getOrCreateContext(doc);
		Element newResourceDSElement = context.addElement("Resource");
		newResourceDSElement.addAttribute("name", amqcf.getJndiName());
		newResourceDSElement.addAttribute("auth", "Container");
		newResourceDSElement.addAttribute("type", TomcatActiveMQConnectionFactory.TYPE);
		newResourceDSElement.addAttribute("factory", amqcf.getFactory());
		newResourceDSElement.addAttribute("brokerName", amqcf.getBrokerName());
		newResourceDSElement.addAttribute("brokerURL", amqcf.getBrokerURL());
		saveDocumentInContextFile(doc);
	}


	public void addOrUpdateResources(List<ResourceLink> resources) {
		Document doc = getDocument();
		for (ResourceLink r : resources) {
			findAndRemoveExistingResourceLink(r, doc);
			Element context = getOrCreateContext(doc);
			Element newResourceDSElement = context.addElement("ResourceLink");
			final CiToXML toXml = new CiToXML(r, newResourceDSElement);
			toXml.addAttribute("type", "java.lang." + r.getType());
			toXml.process();
		}
		saveDocumentInContextFile(doc);
	}

	public void deleteResources(List<ResourceLink> resources) {
		Document doc = getDocument();
		for (ResourceLink r : resources) {
			findAndRemoveExistingResourceLink(r, doc);
		}
		saveDocumentInContextFile(doc);
	}

	private void findAndRemoveExistingResourceLink(ResourceLink r, Document doc) {
		List<Node> resourceTypeList = doc.selectNodes(format("//Context/ResourceLink[@name='%s']", r.getName()));
		for (Node n : resourceTypeList) {
			n.detach();
		}
	}

	public boolean deleteDataSource(TomcatDataSource ds) {
		Document doc = getDocument();
		findAndRemoveExistingDatasource(ds, doc);
		saveDocumentInContextFile(doc);
		return true;
	}

	public boolean deleteTomcatActiveMQConnectionFactory(TomcatActiveMQConnectionFactory amqcf) {
		Document doc = getDocument();
		findAndRemoveExistingAMQCF(amqcf, doc);
		saveDocumentInContextFile(doc);
		return true;
	}

	private void findAndRemoveExistingAMQCF(TomcatActiveMQConnectionFactory amqcf, Document doc) {
		findAndRemoveExistingTomcatResource(amqcf, doc, TomcatActiveMQConnectionFactory.TYPE);
	}

	private void findAndRemoveExistingDatasource(TomcatDataSource ds, Document doc) {
		findAndRemoveExistingTomcatResource(ds, doc, TomcatDataSource.TYPE);
	}

	private Document getDocument() {
		Document doc;
		if (contextFile.exists()) {
			SAXReader reader = new SAXReader();
			try {
				doc = reader.read(contextFile.get());
			} catch (RuntimeIOException e) {
				throw new RuntimeIOException("Unable to parse existing context xml file " + contextFile.getName(), e);
			} catch (DocumentException e) {
				throw new RuntimeIOException("Unable to parse existing context xml file " + contextFile.getName(), e);
			}
		} else {
			doc = DocumentHelper.createDocument();
			//check whether the parent dir exist, in case not create it
			final HostFile parentFile = contextFile.getParentFile();
			if (!parentFile.exists())
				parentFile.mkdirs();
		}
		return doc;
	}

	@SuppressWarnings("rawtypes")
	private void findAndRemoveExistingTomcatResource(TomcatResource tomcatResource, Document doc, String resourceType) {
		List resourceTypeList = doc.selectNodes("//Context/Resource/@type");
		for (Iterator iter = resourceTypeList.iterator(); iter.hasNext();) {
			Attribute type = (Attribute) iter.next();
			String value = type.getValue();
			if (value.equalsIgnoreCase(resourceType)) {
				Element resource = type.getParent();
				Attribute name = (Attribute) resource.selectSingleNode("@name");
				String nameValue = name.getValue();
				if (nameValue.equals(tomcatResource.getJndiName())) {
					Element parent = resource.getParent();
					parent.remove(resource);
				}
			}
		}
	}

	private void findAndRemoveExistingDocBase(Document doc) {
		Attribute docBaseAttr = (Attribute) doc.selectSingleNode("//Context/@docBase");
		if (docBaseAttr != null) {
			docBaseAttr.getParent().remove(docBaseAttr);
		}
	}

	private void saveDocumentInContextFile(Document doc) {
		StringWriter stringWriter = new StringWriter();
		XMLWriter xmlWriter = new XMLWriter(stringWriter);
		try {
			xmlWriter.write(doc);
			xmlWriter.close();
		} catch (IOException e) {
			throw new RuntimeIOException("Unable to add or update Tomcat context xml file " + contextFile.getName(), e);
		}
		HostFileUtils.putStringToHostFile(stringWriter.toString(), contextFile);
	}

	private Element getOrCreateContext(Document doc) {
		Element contextElement;
		if (doc.selectSingleNode("//Context") == null) {
			contextElement = doc.addElement("Context");
		} else {
			contextElement = (Element) doc.selectSingleNode("Context");
		}
		return contextElement;
	}


}
