/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;


import com.xebialabs.deployit.plugin.api.deployment.ResolvedPlaceholder;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.services.SearchParameters;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItemWithPolicies;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployedInfrastructureAsCode;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * A deployment of an application package (a {@link Version}) on an {@link Environment}.
 * <p>
 * Note that this is not the action, but the state of a deployment.
 */
@SuppressWarnings("serial")
@Metadata(description = "A deployment of an application package or a deployable artifact to a middleware CI or an environment.")
@TypeIcon(value = "icons/types/udm.DeployedApplication.svg")
@PublicApiRef
public class DeployedApplication extends BaseConfigurationItemWithPolicies {

    private static final String VERSION_CANDIDATE_FILTER = "deployedApplicationVersionCandidateFilter";

    @Property(description = "The package that was deployed to the environment", candidateValuesFilter = VERSION_CANDIDATE_FILTER)
    @DeployedSpecific
    private Version version;

    @Property(asContainment = true)
    @DeployedSpecific
    private Environment environment;   // same as provisioningEnvironment

    @Property(description = "The items that were deployed to the environment.", required = false)
    @DeployedSpecific
    private Set<Deployed> deployeds = new HashSet<>();

    @Property(description = "The orchestrators that are used to create the deployment plan. Orchestrators will be applied in order.", required = false)
    private List<String> orchestrator;

    @Property(description = "Optimize the generated plan after orchestration.", required = false, defaultValue = "true")
    private boolean optimizePlan;

    @Property(description = "Configuration items resolved from templates on the provisioning package.", required = false, asContainment = false, category = "Provisioning")
    private Set<ConfigurationItem> boundConfigurationItems;

    @Property(description = "Placeholders provided by the user at provisioning.", required = false)
    private Map<String, String> unresolvedPlaceholders = new HashMap<>();

    @Property(description = "Undeploy direct or transient dependencies of this application. A dependent application is undeployed if no other applications depend on it.",
            required = false, defaultValue = "false")
    private boolean undeployDependencies;

    private Set<ResolvedPlaceholder> $resolvedPlaceholders = new HashSet<>();

    public DeployedApplication() {
        // Needed for reflection.
    }

    public DeployedApplication(Version version, Environment environment) {
        this.version = version;
        this.environment = environment;
    }

    @CandidateValuesFilter(name = VERSION_CANDIDATE_FILTER)
    public static SearchParameters findApplicationVersions(ConfigurationItem context, PropertyDescriptor property) {
        DeployedApplication da = (DeployedApplication) context;
        SearchParameters params = new SearchParameters();
        params.setType(property.getReferencedType());
        Version version = da.getVersion();
        if (version != null) {
            params.setParent(version.getApplication().getId());
        }
        return params;
    }

    /**
     * @return The package (a {@link Version}) that is part of this deployment.
     */
    public Version getVersion() {
        return version;
    }

    /**
     * @param version The package (a {@link Version}) that is part of this deployment.
     */
    public void setVersion(Version version) {
        this.version = version;
    }

    /**
     * @return The target {@link Environment} of this deployment.
     */
    public Environment getEnvironment() {
        return environment;
    }

    /**
     * @param environment The target {@link Environment} of this deployment.
     */
    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }

    /**
     * @return The {@link Deployed} items that have been deployed to the {@link Environment}.
     */
    public Set<Deployed> getDeployeds() {
        return deployeds;
    }

    /**
     * @param deployeds The {@link Deployed} items that have been deployed to the {@link Environment}.
     */
    public void setDeployeds(Set<Deployed> deployeds) {
        this.deployeds = deployeds;
    }

    public void addDeployed(Deployed deployed) {
        this.deployeds.add(deployed);
    }

    public void addDeployeds(Deployed... deployeds) {
        this.deployeds.addAll(Arrays.asList(deployeds));
    }

    public void addDeployeds(Collection<Deployed> deployeds) {
        this.deployeds.addAll(deployeds);
    }

    /**
     * @return A list of Orchestrator names that can be used to orchestrate a deployment.
     */
    public List<String> getOrchestrator() {
        return orchestrator;
    }

    public void setOrchestrator(List<String> orchestrator) {
        this.orchestrator = orchestrator;
    }

    public boolean isOptimizePlan() {
        return optimizePlan;
    }

    public void setOptimizePlan(boolean optimizePlan) {
        this.optimizePlan = optimizePlan;
    }

    public Set<ConfigurationItem> getBoundConfigurationItems() {
        return boundConfigurationItems;
    }

    public void setBoundConfigurationItems(Set<ConfigurationItem> boundConfigurationItems) {
        this.boundConfigurationItems = boundConfigurationItems;
    }

    public Map<String, String> getUnresolvedPlaceholders() {
        return unresolvedPlaceholders;
    }

    public void setUnresolvedPlaceholders(Map<String, String> unresolvedPlaceholders) {
        this.unresolvedPlaceholders = unresolvedPlaceholders;
    }

    public Map<String, String> getUnresolvedPlaceholdersWithValues() {
        return getUnresolvedPlaceholders().entrySet().stream().filter(e -> !e.getValue().isEmpty()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    public boolean isUndeployDependencies() {
        return undeployDependencies;
    }

    public void setUndeployDependencies(boolean undeployDependencies) {
        this.undeployDependencies = undeployDependencies;
    }

    public boolean isProvisioning() {
        return (version instanceof DeploymentPackage) && (hasTemplate() || hasProvisionableItem());
    }

    public void add$ResolvedPlaceholder(ResolvedPlaceholder resolvedPlaceholder) {
        $resolvedPlaceholders.add(resolvedPlaceholder);
    }

    public void add$ResolvedPlaceholders(Set<ResolvedPlaceholder> resolvedPlaceholders) {
        $resolvedPlaceholders.addAll(resolvedPlaceholders);
    }

    public Set<ResolvedPlaceholder> get$ResolvedPlaceholders() {
        return $resolvedPlaceholders;
    }

    public void set$resolvedPlaceholders(Set<ResolvedPlaceholder> $resolvedPlaceholders) {
        this.$resolvedPlaceholders = $resolvedPlaceholders;
    }

    public boolean hasProvisioned() {
        return !this.getBoundConfigurationItems().isEmpty() ||
                this.getDeployeds().stream().anyMatch(d -> !d.getBoundConfigurationItems().isEmpty()) ||
                this.getDeployeds().stream().filter(d -> d instanceof BaseDeployedInfrastructureAsCode)
                    .anyMatch(d -> !((BaseDeployedInfrastructureAsCode) d).getGeneratedConfigurationItems().isEmpty());
    }

    private boolean hasTemplate() {
        Predicate<DeploymentPackage> hasBoundTemplate = v -> v.getBoundTemplates() != null && !v.getBoundTemplates().isEmpty();
        Predicate<DeploymentPackage> hasTemplate = v -> v.getTemplates() != null && !v.getTemplates().isEmpty();
        return hasBoundTemplate.or(hasTemplate).test((DeploymentPackage) version);
    }

    private boolean hasProvisionableItem() {
        Predicate<Version> hasProvisionable = v -> v.getDeployables().stream().anyMatch(ci -> ci instanceof Provisionable);
        return Optional
                .ofNullable(version)
                .filter(hasProvisionable)
                .isPresent();
    }
}
