/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.flow;

import com.xebialabs.deployit.local.message.MessageSourceConfigurationHolder;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.services.Repository;

import java.util.Arrays;

import static scala.jdk.javaapi.CollectionConverters.asScala;

public interface ExecutionContext {
    /**
     * Logs a line with new line ending of output. If this step is executing a command, this would be the output that is sent to the {@link System#setOut(java.io.PrintStream)
     * standard output stream}.
     *
     * @param output
     *            the log line to be written.
     */
    void logOutput(String output);

    /**
     * Get the logs line from message handler with the help of key and logs a line. If this step is executing a command, this would be the output that is sent to the {@link System#setOut(java.io.PrintStream)
     * standard output stream}.
     *
     * @param key,
     * @param args sequence of object required.
     */
    void logMsgOutput(String key, Object... args);

    /**
     * Logs a line of output. If this step is executing a command, this would be the output that is sent to the {@link System#setOut(java.io.PrintStream)
     * standard output stream}.
     *
     * @param output
     *            the log line to be written.
     */
    default void logOutputRaw(String output) {
        logOutput(output);
    }


    /**
     * Try to resolve the message. Treat as an error if the message can't be found.
     *
     * @param key  the message code to look up, e.g. 'command.nostep'.
     * @param args an array of arguments that will be filled in for params within
     *             the message (params look like "{0}", "{1,date}", "{2,time}" within a message),
     *             or {@code null} if none
     * @return the resolved message (never {@code null})
     */
    default String getMessage(String key, Object... args) {
        return MessageSourceConfigurationHolder.getMessageService().getMessage(key, asScala(Arrays.asList(args)).toSeq());
    }


    /**
     * Logs a line with a new line ending of error output. If this step is executing a command, this would be the output that is sent to the {@link System#setErr(java.io.PrintStream)
     * standard error stream}.
     *
     * @param error
     *            the error line to be written.
     */
    void logError(String error);

    /**
     * Logs a line of error output. If this step is executing a command, this would be the output that is sent to the {@link System#setErr(java.io.PrintStream)
     * standard error stream}.
     *
     * @param error
     *            the error line to be written.
     */
    default void logErrorRaw(String error) {
        logError(error);
    }

    /**
     * Logs a line with an new line mark ending and an exception to the error output .
     *
     * @param error
     *            the error to be written.
     * @param t
     *            the exception to be logged too
     */
    void logError(String error, Throwable t);

    /**
     * Returns the execution context attribute with the given name, or <tt>null</tt> if there is no attribute by that name.
     *
     * @param name
     *            the name of the attribute
     * @return the value of the attribute, or <tt>null</tt> if there is no attribute by that name.
     */
    Object getAttribute(String name);

    /**
     * Stores an attribute in the execution context. This attribute will be available while the deployment plan is executing, until it is stopped or finished.
     *
     * @param name
     *            the name of the attribute
     * @param value
     *            the value of the attribute
     */
    void setAttribute(String name, Object value);

    /**
     * Returns the repository  where all the ConfigurationItems known in XL Deploy are stored.
     *
     * <em>Note:</em> The repository cannot be stored in a step, as it is not Serializable.
     *
     * @return the repository
     */
    Repository getRepository();


    InspectionContext getInspectionContext();

    /**
     * Returns the task of the current execution context.
     *
     * @return the task
     */
    ITask getTask();
}
