/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;


import java.util.List;
import java.util.Set;

import com.xebialabs.xlplatform.documentation.PublicApiRef;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;

/**
 * A Version of an {@link Application}, which can be represented either as a {@link DeploymentPackage} or as
 * a {@link CompositePackage}.
 * <p>
 * A Version is sometimes also called <i>package</i>, in the sense that it groups together a collection of artifacts and
 * resource specifications of a specific version into a package. This is true especially when considering {@link CompositePackage}s
 * as containing a list of other packages.
 */
@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A version of an application .")
@TypeIcon(value = "icons/types/udm.Version.svg")
@PublicApiRef
public abstract class Version extends BaseConfigurationItem {

    @Property(description = "The application this version belongs to.", asContainment = true)
    private Application application;

    @Property(description = "The orchestrators that are used to create the deployment plan. Orchestrators will be applied in order.", required = false)
    private List<String> orchestrator;

    @Property(hidden = true, defaultValue = "true", description = "Whether to ignore all the undefined properties used in the manifest or to raise an error, while importing a package.", required = false)
    private boolean ignoreUndefinedPropertiesInManifest = true;

    @Property(hidden = true, defaultValue = "false", description = "If this is true, all password properties are exported as plaintext while exporting the package. If false, exportOnlyPasswordPlaceholders is considered.", required = false)
    private boolean exportAllPasswords = false;

    @Property(hidden = true, defaultValue = "true", description = "If this is true, all password properties whose value starts and ends with double curly braces are exported as plaintext. Else, the password values are discarded while exporting the package.", required = false)
    private boolean exportOnlyPasswordPlaceholders = true;

    /**
     * By convention, the version of a Version is the last part of the id (which is in effect the same as the getName() returns.
     *
     * @return The last part of the id
     * @see #getName()
     */
    public String getVersion() {
        return getName();
    }

    /**
     * @return The {@link Application} this package belongs to.
     */
    public Application getApplication() {
        return application;
    }

    /**
     * @param application The {@link Application} this package belongs to.
     */
    public void setApplication(Application application) {
        this.application = application;
    }

    /**
     * @return The list of orchestrator names that are used to deploy the application. Orchestrators will be executed in order.
     */
    public List<String> getOrchestrator() {
        return orchestrator;
    }

    /**
     * @param orchestrator The list of orchestrator names that are used to deploy the application.
     */
    public void setOrchestrator(final List<String> orchestrator) {
        this.orchestrator = orchestrator;
    }

    /**
     * @return The collection of artifacts and resource specifications (a set of {@link Deployable}s).
     */
    public abstract Set<Deployable> getDeployables();
}
