/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.udm;


import java.io.Serializable;
import java.util.*;

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.transformer.PatchDictionary;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.ENVIRONMENTS;

/**
 * An Environment is a grouping of {@link Container} CIs (such as hosts, servers, clusters, etc.) and {@link Dictionary} items.
 * <p>
 * An Environment is used as the target of a deployment, allowing {@link Deployable}s to be mapped to members of
 * the environment.
 */
@SuppressWarnings("serial")
@Metadata(description = "Groups infrastructure CIs and dictionaries and serves as a target for deployment.", root = ENVIRONMENTS)
@TypeIcon(value = "icons/types/udm.Environment.svg")
@PublicApiRef
public class Environment extends BaseConfigurationItem {

    /*
     * Members are not contained in environments.
     */
    @Property(required = false, label = "Containers", description = "The infrastructure components of this Environment")
    private Set<Container> members = new HashSet<>();

    /*
     * Zero or more dictionaries associated with an environment.
     */
    @Property(required = false, description = "The dictionaries providing placeholder values. If the same entry exists in multiple dictionaries, the first one in the list is taken.")
    private List<IDictionary> dictionaries = new ArrayList<>();

    @Property(description = "The directory path where the provisioned environment and provisioned configuration items will be stored.", required = false, category = "Provisioning")
    private String directoryPath = null;

    /*
     * Zero or more patch dictionaries associated with an environment.
     */
    @Property(required = false, description = "List of patch dictionaries that apply changes on the deployables.")
    private List<PatchDictionary> patchDictionaries = new ArrayList<>();

    @Property(required = false, description = "Skip not existing CIs during deployment.", hidden = true)
    private boolean failOnMissingCiRefs = false;

    public Environment() {
        //
    }

    public Environment(Container... members) {
        HashSet<Container> memberSet = new HashSet<>();
        Collections.addAll(memberSet, members);
        setMembers(memberSet);
    }

    /**
     * @return The set of {@link Container}s that are members of this environment.
     */
    public Set<Container> getMembers() {
        return members;
    }

    /**
     * @param classType An instance of the given type.
     * @param <T> The type against which to match the members.
     * @return Only the members that matches the given type.
     */
    @SuppressWarnings("unchecked")
    public <T extends Serializable> Set<T> getMembersOfType(Class<T> classType) {
        Set<T> matchingMembers = new HashSet<>();
        for (Container member : getMembers()) {
            if (classType.isInstance(member)) {
                matchingMembers.add((T) member);
            }
        }
        return matchingMembers;
    }

    /**
     * @param members The set of {@link Container}s that are members of this environment.
     */
    public void setMembers(Set<Container> members) {
        this.members = members;
    }

    /**
     * Adds a {@link Container} as a member of this environment.
     */
    public void addMember(Container member) {
        if (member == null) {
            throw new NullPointerException("Cannot add a null member");
        }
        members.add(member);
    }

    /**
     * The dictionaries providing placeholder values. If the same entry exists in multiple dictionaries,
     * the first one in the list is taken.
     *
     * @return The list of {@link IDictionary}s associated with this environment.
     */
    public List<IDictionary> getDictionaries() {
        return dictionaries;
    }

    /**
     * The dictionaries providing placeholder values. If the same entry exists in multiple dictionaries,
     * the first one in the list is taken.
     *
     * @param dictionaries The list of {@link IDictionary}s associated with this environment.
     */
    public void setDictionaries(List<IDictionary> dictionaries) {
        this.dictionaries = dictionaries;
    }

    public List<PatchDictionary> getPatchDictionaries() {
        return patchDictionaries;
    }

    public void setPatchDictionaries(List<PatchDictionary> patchDictionaries) {
        this.patchDictionaries = patchDictionaries;
    }

    public String getDirectoryPath() {
        return directoryPath;
    }

    public void setDirectoryPath(String directoryPath) {
        this.directoryPath = directoryPath;
    }

    public boolean isFailOnMissingCiRefs() {
        return failOnMissingCiRefs;
    }

    public void setFailOnMissingCiRefs(boolean failOnMissingCiRefs) {
        this.failOnMissingCiRefs = failOnMissingCiRefs;
    }
}
