/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.rules;

import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;

/**
 * Context used for post constructing a {@link com.xebialabs.deployit.plugin.api.flow.Step}. This way a step can receive additional information that can influence it's behavior.
 *
 * A method that is annotated with {@link com.xebialabs.deployit.plugin.api.rules.RulePostConstruct} must accept a single argument of type {@link com.xebialabs.deployit.plugin.api.rules.StepPostConstructContext}.
 */
public class StepPostConstructContext {

    private final Scope scope;

    private final Object scopedObject;

    private final DeployedApplication deployedApplication;

    private final Repository repository;

    public StepPostConstructContext(final Scope scope, final Object scopedObject, final DeployedApplication deployedApplication, final Repository repository) {
        this.scope = scope;
        this.scopedObject = scopedObject;
        this.deployedApplication = deployedApplication;
        this.repository = repository;
    }

    /**
     * @return scope where step will be executed.
     */
    public Scope getScope() {
        return scope;
    }

    /**
     * @return delta or null if scope is not {@link Scope#DEPLOYED}
     */
    public Delta getDelta() {
        return castOrNull(scopedObject, Delta.class);
    }

    /**
     * @return deltas or null if scope is not {@link Scope#PLAN}
     */
    public Deltas getDeltas() {
        return castOrNull(scopedObject, Deltas.class);
    }

    /**
     * @return specification or null if scope is not {@link Scope#PRE_PLAN} or {@link Scope#POST_PLAN}
     */
    public DeltaSpecification getSpecification() {
        return castOrNull(scopedObject, DeltaSpecification.class);
    }

    /**
     * @return application that is being deployed
     */
    public DeployedApplication getDeployedApplication() {
        return deployedApplication;
    }


    /**
     * @return repository
     */
    public Repository getRepository() {
        return repository;
    }

    private <T> T castOrNull(Object object, Class<T> castTo) {
        if(castTo.isInstance(object)) {
            return castTo.cast(object);
        }else {
            return null;
        }
    }
}
