/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.services;

import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 * Declarative query parameters for searching the JCR repository for Configuration Items.
 */
@PublicApiRef
public class SearchParameters {
    Type type;
    String parent;
    String ancestor;
    String name;
    String id;
    Map<String, String> properties = new HashMap<>();
    Calendar before;
    Calendar after;
    long page = 0;
    long resultsPerPage = 0;
    int depth = Integer.MAX_VALUE;
    boolean isExactNameSearch;
    boolean isExactIdSearch;

    /**
     * The desired type of the CI.
     */
    public SearchParameters setType(Type type) {
        this.type = type;
        return this;
    }

    /**
     * Search for the value of a property. This may be a property of type String or Enum.
     * A {@link #type} is required to be able to use this filter.
     */
    public SearchParameters addProperty(String property, String value) {
        if (type == null) {
            throw new IllegalStateException("Cannot search on property value without specifying type.");
        }
        PropertyDescriptor propertyDescriptor = type.getDescriptor().getPropertyDescriptor(property);
        if (propertyDescriptor == null) {
            throw new IllegalArgumentException(String.format("Property '%s' not found on type '%s'.", property, type));
        }
        if (!(propertyDescriptor.getKind() == PropertyKind.STRING) && !(propertyDescriptor.getKind() == PropertyKind.ENUM)) {
            throw new IllegalArgumentException(String.format("Property '%s' on type '%s' is not of type STRING or ENUM.", property, type));
        }
        properties.put(property, value);
        return this;
    }

    /**
     * The desired parent of the CI.
     */
    public SearchParameters setParent(String parent) {
        this.parent = parent;
        return this;
    }

    /**
     * The desired ancestor of the CI.
     */
    public SearchParameters setAncestor(String ancestor) {
        this.ancestor = ancestor;
        return this;
    }

    /**
     * A search pattern for the name. This is like the SQL "LIKE" pattern: the character '%' represents any string of
     * zero or more characters, and the character '_' (underscore) represents any single character.
     */
    public SearchParameters setName(String name) {
        this.name = name;
        return this;
    }

    /**
     * A search pattern for the id. This is like the SQL "LIKE" pattern: the character '%' represents any string of
     * zero or more characters, and the character '_' (underscore) represents any single character.
     */
    public SearchParameters setId(String id) {
        this.id = id;
        return this;
    }

    /**
     * Sets the search parameters to search for CIs modified before the given date.
     */
    public void setBefore(Calendar before) {
        this.before = before;
    }

    /**
     * Sets the search parameters to search for CIs modified after the given date.
     */
    public void setAfter(Calendar after) {
        this.after = after;
    }

    /**
     * Sets the page to be retrieved.
     */
    public SearchParameters setPage(long page) {
        this.page = page;
        return this;
    }

    /**
     * Sets the number of results to be retrieved per page.
     */
    public SearchParameters setResultsPerPage(long nrOfResults) {
        this.resultsPerPage = nrOfResults;
        return this;
    }

    /**
     * Set the depth of the search. By default it searches the full depth of the tree.
     */
    public SearchParameters setDepth(int depth) {
        this.depth = depth;
        return this;
    }
    
    
    /**
     * Set to true for exact name search, default is false.
     */
    public SearchParameters setExactNameSearch(boolean isExactNameSearch) {
        this.isExactNameSearch = isExactNameSearch;
        return this;
    }

    /**
     * Set to true for exact id search, default is false.
     */
    public SearchParameters setExactIdSearch(boolean exactIdSearch) {
        isExactIdSearch = exactIdSearch;
        return this;
    }

    /**
     * @return The CI type.
     */
    public Type getType() {
        return type;
    }

    /**
     * @return  The CI parent id.
     */
    public String getParent() {
        return parent;
    }

    /**
     * @return  The CI ancestor id.
     */
    public String getAncestor() {
        return ancestor;
    }

    /**
     * @return The CI name.
     */
    public String getName() {
        return name;
    }

    /**
     * @return The CI id.
     */
    public String getId() {
        return id;
    }

    public Map<String, String> getProperties() {
        return properties;
    }

    /**
     * @return The specified date for CIs modified before this date.
     */
    public Calendar getBefore() {
        return before;
    }

    /**
     * @return The specified date for CIs modified after this date.
     */
    public Calendar getAfter() {
        return after;
    }

    /**
     * @return The retrieved page.
     */
    public long getPage() {
        return page;
    }

    /**
     * @return The number of results retrieved per page.
     */
    public long getResultsPerPage() {
        return resultsPerPage;
    }

    /**
     * @return The depth of the search. Default is full depth.
     */
    public int getDepth() {
        return depth;
    }

    /**
     * @return  true for exactNameSearch
     */
    public boolean isExactNameSearch() {
        return isExactNameSearch;
    }

    /**
     * @return  true for exactIdSearch
     */
    public boolean isExactIdSearch() {
        return isExactIdSearch;
    }

    @Override
    public String toString() {
        return String.format("SearchParameters [type=%s, properties=%s, parent=%s, ancestor=%s, name=%s, id=%s, before=%s, after=%s, page=%d, resultsPerPage=%d, depth = %d]",
                type, properties, parent, ancestor, name, id, before, after, page, resultsPerPage, depth);
    }

}
