/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.plugin.api.reflect;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

/**
 * Encapsulates the metadata about a property of a type in the XL Deploy type system. Example of such metadata:
 * the name and type of the property, information related to UI-rendering etc.
 */
@PublicApiRef
public interface PropertyDescriptor {

    /**
     * @return The name of the property.
     */
    String getName();

    /**
     * @return Textual description of the property.
     */
    String getDescription();

    /**
     * Indicates a parent / child relationship (Boolean). Only applicable for properties that refer to other CIs.
     *
     * @return If asContainment is set to true and the property is of kind CI then we are referring to the parent of
     * this CI. If the kind is LIST_OF_CI or SET_OF_CI, then this CI is the parent and this property refers to its children.
     */
    boolean isAsContainment();

    /**
     * Indicates a child relationship (Boolean). Only applicable to properties of kind CI.
     *
     * @return If child is set to true and the property is of kind CI then we are referring to the child of
     * this CI.
     */
    boolean isNested();

    int getOrder();

    /**
     * The UI groups all properties in the same category into a tab of their own.
     *
     * @return The category of this property.
     */
    String getCategory();

    /**
     * @return Label text for use in the UI.
     */
    String getLabel();

    /**
     * Only applicable for String property types. Contents maybe masked with '*' character.
     *
     * @return Whether this property is a password.
     */
    boolean isPassword();

    /**
     * @return Whether this property is required.
     */
    boolean isRequired();

    /**
     * @return The size of the input field in the UI. Possible values: DEFAULT, SMALL, MEDIUM or LARGE.
     */
    Property.Size getSize();

    /**
     * @return The {@link com.xebialabs.deployit.plugin.api.reflect.PropertyKind}. Possible values: BOOLEAN, INTEGER, STRING, ENUM, CI, SET_OF_STRING, SET_OF_CI,
     * LIST_OF_STRING, LIST_OF_CI or MAP_STRING_STRING.
     */
    PropertyKind getKind();

    /**
     * @return All the available values if this property is of kind ENUM. Otherwise, an empty list.
     */
    List<String> getEnumValues();

    /**
     * Only relevant for properties of kind CI, SET_OF_CI or LIST_OF_CI.
     *
     * @return The type of the referenced configuration item.
     */
    Type getReferencedType();

    /**
     * @return The default value of the property.
     */
    Object getDefaultValue();

    DescriptorRegistryId getTypeSource();

    /**
     * Hidden properties don't show up in the UI. Note that a hidden property must have a default value.
     *
     * @return Whether the property is hidden.
     */
    boolean isHidden();

    /**
     * @param item A {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     * @return The value for this property of the given CI.
     */
    Object get(ConfigurationItem item);

    /**
     * Set the given value for this property of the given {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}.
     */
    void set(ConfigurationItem item, Object value);

    boolean areEqual(ConfigurationItem item, ConfigurationItem other);

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Function<ConfigurationItem, Object> identifierExtractor);

    /**
     * @return Whether this property is used for inspection (discovery).
     */
    boolean isInspectionProperty();

    /**
     * @return Whether this property is required in order to start the discovery process.
     */
    boolean isRequiredForInspection();

    /**
     * @return Whether the property is persisted in the repository or not.
     */
    boolean isTransient();

    /**
     * @return Whether the property is readonly or not.
     */
    boolean isReadonly();

    /**
     * @return The aliases with which this property can be referred.
     */
    Set<String> getAliases();

    /**
     * The {@link com.xebialabs.deployit.plugin.api.udm.CandidateValuesFilter} to apply when selecting values for this CI, SET_OF_CI, LIST_OF_CI property
     *
     * @return The name of the {@link com.xebialabs.deployit.plugin.api.udm.CandidateValuesFilter}
     */
    String getCandidateValuesFilter();

    /**
     * @return The fully qualified name of the property. This is the type name followed by a dot and then the property name.
     * For example: <code>overthere.SshJumpstation.port</code>.
     */
    String getFqn();

    boolean isDeployedSpecific();

    /**
     * @return The {@link InputHint} configured for this property
     */
    InputHint getInputHint();

    List<Annotation> getAnnotations();

    @Override
    String toString();
}
