package com.xebialabs.deployit.plugin.api.deployment.execution;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;

import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.sort;
import static java.util.Collections.unmodifiableList;

/**
 * The lowest type of sub-plan, this plan contains {@link DeploymentStep}s which can be executed and {@link Delta} objects
 * for which this InterleavedPlan has been built.
 *
 * The {@link DeploymentStep}s in the InterleavedPlan are sorted by the {@link StepComparator}
 */
public class InterleavedPlan implements Plan {
    private List<Delta> deltas;
    private List<DeploymentStep> steps;

    InterleavedPlan(Delta... deltas) {
        this.deltas = unmodifiableList(newArrayList(deltas));
        this.steps = unmodifiableList(Lists.<DeploymentStep>newArrayList());
    }

	/**
	 * Constructor. Takes the steps, and sorts them in a new List, the passed in list is not structurally modified.
	 * Steps in the InterlavedPlan are sorted using the {@link StepComparator}
	 * @param deltas
	 * @param steps
	 */
    InterleavedPlan(List<Delta> deltas, List<DeploymentStep> steps) {
        this.deltas = unmodifiableList(deltas);
        sort(steps, new StepComparator());
        this.steps = unmodifiableList(steps);
    }

	/**
	 * Gets the {@link Delta}s for which this InterleavedPlan has been compiled
	 * @return the Delta's
	 */
    public List<Delta> getDeltas() {
        return deltas;
    }

	@Override
    public List<DeploymentStep> getSteps() {
        return steps;
    }

    @Override
    public String getType() {
        return InterleavedPlan.class.getName();
    }
}
