/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.mapper;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ResolutionException;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.Environment;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.mapper.Pair;
import com.xebialabs.deployit.plugin.was.ci.WarsWebserversVirtualHostMapping;
import com.xebialabs.deployit.plugin.was.ci.WasEarMapping;
import com.xebialabs.deployit.plugin.was.ci.WasFileServing;
import com.xebialabs.deployit.plugin.was.ci.WasManagedApacheHttpdServer;
import com.xebialabs.deployit.plugin.was.ci.WasNodeAgent;
import com.xebialabs.deployit.plugin.was.ci.WasTarget;
import com.xebialabs.deployit.plugin.was.ci.WasUnManagedApacheHttpdServer;
import com.xebialabs.deployit.plugin.was.ci.WasWarMapping;
import com.xebialabs.deployit.plugin.was.step.CopyWasWebServerPluginConfigurationStep;
import com.xebialabs.deployit.plugin.was.step.CreateWasVirtualHostStep;
import com.xebialabs.deployit.plugin.was.step.DestroyWasVirtualHostStep;
import com.xebialabs.deployit.plugin.was.step.GenerateWasUnManagedWebServerPluginConfigurationStep;
import com.xebialabs.deployit.plugin.was.step.SynchronizeWasNodeStep;
import com.xebialabs.deployit.plugin.was.step.UpdateWasFileServingEnabledStep;
import com.xebialabs.deployit.plugin.was.step.UpdateWasVirtualHostAliasesStep;

public abstract class JeeWebArtifactToWasTargetMapper<S extends NamedDeployableArtifact, M extends Mapping<? super S, ? super T>, T extends WasTarget> extends
        WasModificationSupportingStepGeneratingMapper<S, M, T> {

	public JeeWebArtifactToWasTargetMapper(Change<Deployment> change, boolean applyDefaultMapping) {
		super(change, applyDefaultMapping);
	}

	@Override
	protected void generateAdditionStepsForAddedMapping(S ear, M mapping, T target, List<Step> steps) {
		Set<WasManagedApacheHttpdServer> webservers = getWebservers(mapping);

		String virtualHostAlias = validateAndGetVirtualHostFromMapping(mapping);
		boolean isAppVirtualHostRequired = getAppVirtualHostRequired(mapping);
		if (isAppVirtualHostRequired) {
			steps.add(new CreateWasVirtualHostStep(target.getCell(), ear.getName()));
			steps.add(new UpdateWasVirtualHostAliasesStep(target.getCell(), ear.getName(), virtualHostAlias));
		}

		// for each Wars inside the Ear, create virtual host and update the aliases
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelVirtualHostMappingPresent()) {
				steps.add(new CreateWasVirtualHostStep(target.getCell(), getVirtualHostFromMappingOrDefault(eachMapping, ear.getName())));
				steps.add(new UpdateWasVirtualHostAliasesStep(target.getCell(), getVirtualHostFromMappingOrDefault(eachMapping, ear.getName()),
				        getVirtualHostAliasFromMappingOrDefault(eachMapping, virtualHostAlias)));
			}
		}

		generateDeployStep(ear, mapping, target, webservers, steps);

		// for each Wars inside the Ear , Update File Serving Enabled.
		List<WarsWebserversVirtualHostMapping> fileServingEnabledPerWars = getFileServingEnabledPerWars(mapping);
		for (WarsWebserversVirtualHostMapping warsWebserversVHMapping : fileServingEnabledPerWars) {
			if (warsWebserversVHMapping.getFileServing() != null && warsWebserversVHMapping.getFileServing() != WasFileServing.DO_NOT_OVERRIDE) {
				steps.add(new UpdateWasFileServingEnabledStep(ear, target, warsWebserversVHMapping.getWarName(), warsWebserversVHMapping.getFileServing()));
			}
		}

		if (isAppVirtualHostRequired) {
			associateVirtualHostWithWebServers(webservers, virtualHostAlias);
		}

		// for each War present in the Ear
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getNewRevision().getTarget(), eachMapping
				        .getWebserver());
				associateVirtualHostWithWebServers(webServersFromEnvironment, getVirtualHostAliasFromMappingOrDefault(eachMapping, virtualHostAlias));
			}
		}

	}

	protected void associateVirtualHostWithWebServers(Set<WasManagedApacheHttpdServer> webservers, String virtualHostAlias) {
		for (WasManagedApacheHttpdServer eachWebServer : webservers) {
			associateVirtualHostWithWebServer(virtualHostsPerWebServer, eachWebServer, virtualHostAlias);
		}
	}

	@Override
	protected void generateDeletionStepsForDeletedMapping(S ear, M mapping, T target, List<Step> steps) {
		Set<WasManagedApacheHttpdServer> webservers = getWebservers(mapping);

		generateStopStep(ear, target, mapping, steps);
		generateUndeployStep(ear, target, mapping, steps);
		boolean isAppVirtualHostRequired = getAppVirtualHostRequired(mapping);
		if (isAppVirtualHostRequired) {
			steps.add(new DestroyWasVirtualHostStep(target.getCell(), ear.getName()));
		}
		String virtualHostAlias = validateAndGetVirtualHostFromMapping(mapping);

		// for each War present in the Ear file
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelVirtualHostMappingPresent()) {
				steps.add(new DestroyWasVirtualHostStep(target.getCell(), getVirtualHostFromMappingOrDefault(eachMapping, ear.getName())));
			}
		}

		if (isAppVirtualHostRequired) {
			associateVirtualHostWithWebServers(webservers, virtualHostAlias);
		}

		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getOldRevision().getTarget(), eachMapping
				        .getWebserver());
				associateVirtualHostWithWebServers(webServersFromEnvironment, getVirtualHostAliasFromMappingOrDefault(eachMapping, virtualHostAlias));
			}
		}
	}

	@SuppressWarnings("unchecked")
	public void generateStartSteps(List<Step> steps) {
		for (Pair<M, M> m : modifiedMappings) {
			M newM = m.getSecond();
			generateStartStep((S) newM.getSource(), (T) newM.getTarget(), newM, steps);
		}

		for (M m : addedMappings) {
			generateStartStep((S) m.getSource(), (T) m.getTarget(), m, steps);
		}
	}

	@Override
	protected void generateModificationStepsForModifiedMapping(S oldMappingSource, M oldVersionOfModifiedMapping, T oldMappingTarget, S newMappingSource,
	        M newVersionOfModifiedMapping, T newMappingTarget, List<Step> steps) {

		String oldVirtualHost = null;
		String newVirtualHost = null;
		if (WasEarMapping.class.isInstance(oldVersionOfModifiedMapping)) {
			oldVirtualHost = WasEarMapping.class.cast(oldVersionOfModifiedMapping).getVirtualHost();
			newVirtualHost = WasEarMapping.class.cast(newVersionOfModifiedMapping).getVirtualHost();
		} else {
			oldVirtualHost = WasWarMapping.class.cast(oldVersionOfModifiedMapping).getVirtualHost();
			newVirtualHost = WasWarMapping.class.cast(newVersionOfModifiedMapping).getVirtualHost();
		}

		boolean mappingSourceChanged = !oldMappingSource.equals(newMappingSource);
		boolean virtualHostChanged = !oldVirtualHost.equals(newVirtualHost);
		boolean virtualHostChangedForWarsInsideEar = false;

		// delete steps
		Set<WasManagedApacheHttpdServer> oldWebservers = getWebservers(oldVersionOfModifiedMapping);

		generateStopStep(oldMappingSource, oldMappingTarget, oldVersionOfModifiedMapping, steps);
		if (mappingSourceChanged) {
			generateUndeployStep(oldMappingSource, oldMappingTarget, oldVersionOfModifiedMapping, steps);
		}

		if (virtualHostChanged) {
			steps.add(new DestroyWasVirtualHostStep(oldMappingTarget.getCell(), oldMappingSource.getName()));
		}

		// DestroyWasVirtualHostStep for the virtual hosts of individual Wars inside Ear
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(oldVersionOfModifiedMapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelVirtualHostMappingPresent()) {
				virtualHostChangedForWarsInsideEar = true;
				steps
				        .add(new DestroyWasVirtualHostStep(oldMappingTarget.getCell(), getVirtualHostFromMappingOrDefault(eachMapping, oldMappingSource
				                .getName())));
			}
		}

		String oldVirtualHostAlias = validateAndGetVirtualHostFromMapping(oldVersionOfModifiedMapping);
		associateVirtualHostWithWebServers(oldWebservers, oldVirtualHostAlias);

		// associate step for the virtual hosts of individual Wars inside Ear
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getNewRevision().getTarget(), eachMapping
				        .getWebserver());
				associateVirtualHostWithWebServers(webServersFromEnvironment, getVirtualHostAliasFromMappingOrDefault(eachMapping, oldVirtualHostAlias));
			}
		}

		// Synch the nodes
		if (virtualHostChanged || virtualHostChangedForWarsInsideEar) {
			for (WasNodeAgent eachNode : oldMappingTarget.getNodes()) {
				steps.add(new SynchronizeWasNodeStep(eachNode));
			}
		}

		// addition steps
		if (virtualHostChanged) {
			steps.add(new CreateWasVirtualHostStep(newMappingTarget.getCell(), newMappingSource.getName()));
			steps.add(new UpdateWasVirtualHostAliasesStep(newMappingTarget.getCell(), newMappingSource.getName(),
			        validateAndGetVirtualHostFromMapping(newVersionOfModifiedMapping)));
		}
		// addition steps for the Wars inside the Ear
		List<WarsWebserversVirtualHostMapping> newWebserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(newVersionOfModifiedMapping);
		for (WarsWebserversVirtualHostMapping eachMapping : newWebserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelVirtualHostMappingPresent()) {
				virtualHostChangedForWarsInsideEar = true;
				steps
				        .add(new CreateWasVirtualHostStep(newMappingTarget.getCell(), getVirtualHostFromMappingOrDefault(eachMapping, newMappingSource
				                .getName())));
				steps.add(new UpdateWasVirtualHostAliasesStep(newMappingTarget.getCell(), getVirtualHostFromMappingOrDefault(eachMapping, newMappingSource
				        .getName()), getVirtualHostAliasFromMappingOrDefault(eachMapping, newVirtualHost)));
			}
		}

		Set<WasManagedApacheHttpdServer> newWebservers = getWebservers(newVersionOfModifiedMapping);
		String newVirtualHostAlias = validateAndGetVirtualHostFromMapping(newVersionOfModifiedMapping);
		associateVirtualHostWithWebServers(newWebservers, newVirtualHostAlias);

		// association step for each War inside the Ear
		for (WarsWebserversVirtualHostMapping eachMapping : newWebserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getNewRevision().getTarget(), eachMapping
				        .getWebserver());
				associateVirtualHostWithWebServers(webServersFromEnvironment, getVirtualHostAliasFromMappingOrDefault(eachMapping, newVirtualHostAlias));
			}
		}

		if (mappingSourceChanged) {
			generateDeployStep(newMappingSource, newVersionOfModifiedMapping, newMappingTarget, newWebservers, steps);
			List<WarsWebserversVirtualHostMapping> fileServingEnabledPerWars = getFileServingEnabledPerWars(newVersionOfModifiedMapping);
			for (WarsWebserversVirtualHostMapping warsWebserversVHMapping : fileServingEnabledPerWars) {
				if (warsWebserversVHMapping.getFileServing() != null && warsWebserversVHMapping.getFileServing() != WasFileServing.DO_NOT_OVERRIDE) {
					steps.add(new UpdateWasFileServingEnabledStep(newMappingSource, newMappingTarget, warsWebserversVHMapping.getWarName(),
					        warsWebserversVHMapping.getFileServing()));
				}
			}
		}

		// synch the nodes
		if (virtualHostChanged || virtualHostChangedForWarsInsideEar) {
			for (WasNodeAgent eachNode : newMappingTarget.getNodes()) {
				steps.add(new SynchronizeWasNodeStep(eachNode));
			}
		}

	}

	public void generateAndCopyPluginSteps(List<Step> steps) {
		Set<WasUnManagedApacheHttpdServer> webservers = new HashSet<WasUnManagedApacheHttpdServer>();

		for (M m : deletedMappings) {
			webservers.addAll(getUnManagedWebservers(m));
		}

		for (Pair<M, M> m : modifiedMappings) {
			M newM = m.getSecond();
			webservers.addAll(getUnManagedWebservers(newM));
		}

		for (M m : addedMappings) {
			webservers.addAll(getUnManagedWebservers(m));
		}

		for (WasUnManagedApacheHttpdServer wasUnManagedApacheHttpdServer : webservers) {
			generateAndCopyPluginSteps(wasUnManagedApacheHttpdServer, steps);
		}
	}

	/*
	 * "de-generification" required to get round a strange compilation error, i.e. using an instance method with type parameter M doesn't work here (?!?).
	 */
	protected static Set<WasManagedApacheHttpdServer> getWebservers(Mapping<?, ?> mapping) {
		Set<WasManagedApacheHttpdServer> webservers;
		if (mapping instanceof WasEarMapping) {
			webservers = ((WasEarMapping) mapping).getWebservers();
		} else if (mapping instanceof WasWarMapping) {
			webservers = ((WasWarMapping) mapping).getWebservers();
		} else {
			webservers = Collections.emptySet();
		}
		return webservers;
	}

	protected Set<WasUnManagedApacheHttpdServer> getUnManagedWebservers(M mapping) {
		Set<WasUnManagedApacheHttpdServer> unManagedWebservers = Sets.newHashSet();
		Set<WasManagedApacheHttpdServer> allWebservers = getWebservers(mapping);
		for (WasManagedApacheHttpdServer apacheHttpdServer : allWebservers) {
			if (apacheHttpdServer instanceof WasUnManagedApacheHttpdServer) {
				unManagedWebservers.add((WasUnManagedApacheHttpdServer) apacheHttpdServer);
			}
		}
		List<WarsWebserversVirtualHostMapping> newWebserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : newWebserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Deployment deployment = change.getNewRevision();
				if (deployment == null) {
					deployment = change.getOldRevision();
				}
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(deployment.getTarget(), eachMapping.getWebserver());
				for (WasManagedApacheHttpdServer wasManagedApacheHttpdServer : webServersFromEnvironment) {
					if (wasManagedApacheHttpdServer instanceof WasUnManagedApacheHttpdServer) {
						unManagedWebservers.add((WasUnManagedApacheHttpdServer) wasManagedApacheHttpdServer);
					}
				}
			}
		}
		return unManagedWebservers;
	}

	protected String validateAndGetVirtualHostFromMapping(M mapping) {
		String vhost = getVirtualHostFromMapping(mapping);
		if (StringUtils.isBlank(vhost)) {
			throw new ResolutionException("Property \"virtualHost\" is not set on mapping " + mapping);
		}
		return vhost;
	}

	protected String getVirtualHostAliasFromMappingOrDefault(WarsWebserversVirtualHostMapping eachMapping, String defaultVirtualHostAlias) {
		if (StringUtils.isBlank(eachMapping.getVirtualHostAlias())) {
			return defaultVirtualHostAlias;
		}
		return eachMapping.getVirtualHostAlias();
	}

	protected String getVirtualHostFromMappingOrDefault(WarsWebserversVirtualHostMapping eachMapping, String defaultVirtualHost) {
		if (StringUtils.isBlank(eachMapping.getVirtualHost())) {
			return defaultVirtualHost;
		}
		return eachMapping.getVirtualHost();
	}

	protected Set<WasManagedApacheHttpdServer> getWebServerFromEnvironment(Environment env, String webServerNames) {
		Set<WasManagedApacheHttpdServer> webserversFromEnvironment = Sets.newHashSet();
		String[] serversArray = StringUtils.split(webServerNames, ";");
		for (String serverNameFromMapping : serversArray) {
			WasManagedApacheHttpdServer webserverFromEnvironment = getWebserverFromEnvironmentWithLabel(env, serverNameFromMapping);
			webserversFromEnvironment.add(webserverFromEnvironment);
		}
		return webserversFromEnvironment;
	}

	private WasManagedApacheHttpdServer getWebserverFromEnvironmentWithLabel(Environment env, String webserverName) {
		Set<WasManagedApacheHttpdServer> webserversInEnvironment = env.getMembersOfType(WasManagedApacheHttpdServer.class);
		for (WasManagedApacheHttpdServer eachServer : webserversInEnvironment) {
			if (eachServer.getLabel().equals(webserverName)) {
				return eachServer;
			}
		}
		throw new ResolutionException("No webserver with label '" + webserverName + "' found in the environment");
	}

	protected void generateAndCopyPluginSteps(WasUnManagedApacheHttpdServer webserver, List<Step> steps) {
		steps.add(new GenerateWasUnManagedWebServerPluginConfigurationStep(webserver));
		steps.add(new CopyWasWebServerPluginConfigurationStep(webserver));
	}

	protected List<WarsWebserversVirtualHostMapping> getUpdatedWebserversMapping(M mapping) {
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (eachMapping.isWarLevelMappingPresent()) {
				Set<WasManagedApacheHttpdServer> webServersFromEnvironment = getWebServerFromEnvironment(change.getNewRevision().getTarget(), eachMapping
				        .getWebserver());
				eachMapping.setWebservers(webServersFromEnvironment);
			}
		}
		return webserversAndVirtualHostPerWars;
	}

	protected boolean getAppVirtualHostRequired(M mapping) {
		boolean mappingRequired = true;
		List<WarsWebserversVirtualHostMapping> webserversAndVirtualHostPerWars = getWebserversAndVirtualHostPerWar(mapping);
		for (WarsWebserversVirtualHostMapping eachMapping : webserversAndVirtualHostPerWars) {
			if (!eachMapping.isWarLevelVirtualHostMappingPresent()) {
				return true;
			} else {
				mappingRequired = false;
			}
		}
		return mappingRequired;
	}

	protected abstract void generateStartStep(S artifact, T target, M mapping, List<Step> steps);

	protected abstract void generateStopStep(S artifact, T target, M mapping, List<Step> steps);

	protected abstract String getVirtualHostFromMapping(M mapping);

	protected abstract List<WarsWebserversVirtualHostMapping> getWebserversAndVirtualHostPerWar(M mapping);

	protected abstract List<WarsWebserversVirtualHostMapping> getFileServingEnabledPerWars(M mapping);

	protected abstract void generateDeployStep(S artifact, M mapping, T target, Collection<WasManagedApacheHttpdServer> webservers, List<Step> steps);

	protected abstract void generateUndeployStep(S artifact, T target, M mapping, List<Step> steps);

}
