/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.mapper;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.plugin.wls.ci.WlsArtifactMapping;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.ci.WlsStageMode;
import com.xebialabs.deployit.plugin.wls.ci.WlsTarget;
import com.xebialabs.deployit.plugin.wls.step.WlsDeployArtifactStep;
import com.xebialabs.deployit.plugin.wls.step.WlsDeployNoStageArtifactStep;
import com.xebialabs.deployit.steps.CopyStep;

/**
 * Helper class handling the generation of Wls Deployment Steps using the stage/no stage mode and deployment plan.
 */
public class WlsStageModeMapperHelper {

    public List<? extends Step> generateDeploySteps(List<WlsArtifactMapping> mappings) {
        List<Step> stageSteps = new LinkedList<Step>();

        List<Step> noStageSteps = new LinkedList<Step>();
        Set<CopyStep> copySteps = new HashSet<CopyStep>();

        boolean noStageModeFound = false;
        for (WlsArtifactMapping m : mappings) {
            noStageModeFound = isNoStageMode(m);
            if (!noStageModeFound) {
                //stageMode
                WlsDeployArtifactStep step = new WlsDeployArtifactStep(m.getTarget(), m.getSource());

                if (m.getDeploymentPlan() != null) {
                    Host activeHost = m.getTarget().getDomain().getActiveHost();
                    String remoteDeploymentPlanDestination = m.getDeploymentPlanStagingDirectory() + activeHost + m.getSource().getName() + ".xml";
                    if (StringUtils.isBlank(m.getDeploymentPlanStagingDirectory()))
                        throw new IllegalArgumentException("With a deployment plan, a deployment plan staging directory must be filled.");

                    step.setDeploymentPlan(m.getDeploymentPlan(), remoteDeploymentPlanDestination);
                }

                stageSteps.add(step);

            } else {
                //noStageMode
                String remoteDestination = "";
                String remoteDeploymentPlanDestination = "";

                for (Host remoteHost : findHosts(m.getTarget())) {
                    String separator = remoteHost.getFileSeparator();
                    if (StringUtils.isBlank(m.getStagingDirectory()))
                        throw new IllegalArgumentException("With noStage Mode, a staging directory must be filled.");

                    remoteDestination = m.getStagingDirectory() + separator + m.getSource().getName() + "." + getExtension(m.getSource());
                    copySteps.add(new CopyStep(Host.getLocalHost(), m.getSource().getLocation(), remoteHost, remoteDestination, true));
                    if (m.getDeploymentPlan() != null) {
                        if (StringUtils.isBlank(m.getDeploymentPlanStagingDirectory()))
                            throw new IllegalArgumentException("With a deployment plan, a deployment plan staging directory must be filled.");

                        remoteDeploymentPlanDestination = m.getDeploymentPlanStagingDirectory() + separator + m.getSource().getName() + ".xml";
                        copySteps.add(new CopyStep(Host.getLocalHost(), m.getDeploymentPlan().getLocation(), remoteHost, remoteDeploymentPlanDestination, true));
                    }
                }

                WlsDeployNoStageArtifactStep step = new WlsDeployNoStageArtifactStep(m.getTarget(), m.getSource(), remoteDestination);
                if (m.getDeploymentPlan() != null) {
                    step.setRemoteDeploymentPlanPath(remoteDeploymentPlanDestination);
                }
                noStageSteps.add(step);

            }
        }


        //Merge all steps;
        List<Step> allSteps = new ArrayList<Step>();
        allSteps.addAll(stageSteps);
        allSteps.addAll(copySteps);
        allSteps.addAll(noStageSteps);

        return allSteps;
    }

    private String getExtension(NamedDeployableArtifact artifact) {
        try {
            final Field field = artifact.getClass().getField("ARCHIVE_EXTENSION");
            return "." + field.get(artifact).toString();
        } catch (Exception e) {
            throw new RuntimeException("Unexpected artifact " + artifact, e);
        }
    }

    private Set<Host> findHosts(WlsTarget target) {

        Set<Host> hosts = new HashSet<Host>();

        if (target instanceof WlsServer) {
            WlsServer server = (WlsServer) target;
            hosts.add(server.getHost());
        }

        if (target instanceof WlsCluster) {
            WlsCluster cluster = (WlsCluster) target;
            for (WlsServer server : cluster.getServers())
                hosts.add(server.getHost());
        }

        return hosts;
    }


    private boolean isNoStageMode(WlsArtifactMapping artefactMapping) {
        final WlsTarget target = artefactMapping.getTarget();
        if (target instanceof WlsCluster) {
            return artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
        }
        if (target instanceof WlsServer) {
            WlsServer server = (WlsServer) target;
            return server.getStageMode().equals(WlsStageMode.NoStage) || artefactMapping.getStageMode().equals(WlsStageMode.NoStage);

        }
        return artefactMapping.getStageMode().equals(WlsStageMode.NoStage);
    }
}

