/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.step;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.mapping.PlaceholderFormat;
import com.xebialabs.deployit.hostsession.*;
import com.xebialabs.deployit.plugin.wls.ci.WlsCluster;
import com.xebialabs.deployit.plugin.wls.ci.WlsDeploymentPlan;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.ci.WlsTarget;
import org.apache.commons.lang.StringUtils;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Deploys applications on a {@link WlsCluster cluster} or {@link WlsServer managed server}.
 */
@SuppressWarnings("serial")
public class WlsDeployArtifactStep extends WlsStepBase {

	private static final String DEPLOY_APPLICATION_ON_WEBLOGIC_SCRIPT_NAME = STEP_RESOURCES_PATH + "deploy-application.py";

	private static final String DEPLOY_APPLICATION_ON_WEBLOGIC_WITH_PLAN_SCRIPT_NAME = STEP_RESOURCES_PATH + "deploy-plan-application.py";

	private String targetName;

	private String applicationName;

	private String extension;

	private String applicationLocation;

	private String deploymentPlanLocation;

	private String remoteDeploymentPlanDestination;

	private Map<String, String> properties;

	private PlaceholderFormat placeholderFormat;

	public WlsDeployArtifactStep(WlsTarget target, NamedDeployableArtifact artifact) {
		this(target, artifact, Collections.<String, String>emptyMap(), PlaceholderFormat.SPRING);
	}

	public WlsDeployArtifactStep(WlsTarget target, NamedDeployableArtifact artifact, Map<String, String> properties, PlaceholderFormat placeholderFormat) {
		super(target.getDomain());
		this.targetName = target.getName();
		this.applicationName = artifact.getName();
		this.extension = findOutExtension(artifact);
		this.applicationLocation = artifact.getLocation();
		this.properties = properties;
		this.placeholderFormat = placeholderFormat;

		if (properties == null || properties.isEmpty())
			setDescription("Deploy " + artifact + " on " + target);
		else
			setDescription("Deploy " + artifact + " on " + target + " while replacing properties in the archive.");
	}


	@Override
	public String getDescription() {
		String desc = super.getDescription();
		if (hasDeploymentPlan()) {
			desc = desc + " and a deployment plan";
		}
		return desc;
	}

	private String findOutExtension(NamedDeployableArtifact artifact) {
		try {
			final Field field = artifact.getClass().getField("ARCHIVE_EXTENSION");
			return "." + field.get(artifact).toString();
		} catch (Exception e) {
			throw new RuntimeException("Unexpected artifact " + artifact, e);
		}
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession rhs = domain.connectToAdminHost();
		try {
			HostFile uploadedApplication = uploadApplication(rhs);
			String uploadedApplicationPath = uploadedApplication.getPath().replace('\\', '/');

			List<String> argsList = new ArrayList<String>();
			argsList.add(domain.getUsername());
			argsList.add(domain.getPassword());
			argsList.add(domain.getConnectUrl());
			argsList.add(targetName);
			argsList.add(applicationName);
			argsList.add(uploadedApplicationPath);

			String resourcePath = DEPLOY_APPLICATION_ON_WEBLOGIC_SCRIPT_NAME;
			if (hasDeploymentPlan()) {
				resourcePath = DEPLOY_APPLICATION_ON_WEBLOGIC_WITH_PLAN_SCRIPT_NAME;
				HostFile uploadedPan = uploadPlan(rhs);
				String uploadedPlanPath = uploadedPan.getPath().replace('\\', '/');
				argsList.add(uploadedPlanPath);
			}
			String[] args = argsList.toArray(new String[argsList.size()]);

			int res = executeWlstJythonScriptResource(ctx, rhs, resourcePath, args);
			return res == 0;
		} finally {
			rhs.close();
		}
	}

	private HostFile uploadPlan(HostSession rhs) {
		HostSession lhs = Host.getLocalHost().getHostSession();
		try {
			HostFile planFileToUpload = lhs.getFile(deploymentPlanLocation);
			HostFile uploadedPlanFile = rhs.getFile(remoteDeploymentPlanDestination);
			HostFile parentFile = uploadedPlanFile.getParentFile();
			if (!parentFile.exists())
				parentFile.mkdirs();
			HostFileUtils.copy(planFileToUpload, uploadedPlanFile, getTransformer());
			return uploadedPlanFile;
		} finally {
			lhs.close();
		}
	}

	private HostFile uploadApplication(HostSession rhs) {
		HostSession lhs = Host.getLocalHost().getHostSession();
		try {
			HostFile applicationFileToUpload = lhs.getFile(applicationLocation);
			HostFile uploadedApplicationFile = rhs.getTempFile(applicationName, extension);
			HostFileUtils.copy(applicationFileToUpload, uploadedApplicationFile, getTransformer());
			return uploadedApplicationFile;
		} finally {
			lhs.close();
		}
	}

	private HostFileInputStreamTransformer getTransformer() {
		if (properties.isEmpty())
			return null;
		return new LenientTemplateResolvingArchiveHostFileInputTransformer(properties, placeholderFormat);
	}

	public String getApplicationName() {
		return applicationName;
	}

	public String getExtension() {
		return extension;
	}

	public String getApplicationLocation() {
		return applicationLocation;
	}

	public String getTargetName() {
		return targetName;
	}

	public void setDeploymentPlan(WlsDeploymentPlan plan, String remoteDeploymentPlanDestination) {
		this.deploymentPlanLocation = plan.getLocation();
		this.remoteDeploymentPlanDestination = remoteDeploymentPlanDestination;
	}

	public boolean hasDeploymentPlan() {
		return !StringUtils.isBlank(deploymentPlanLocation) && !StringUtils.isBlank(remoteDeploymentPlanDestination);
	}


}
