package com.xebialabs.deployit.repository.sql.artifacts.mover

import java.util.concurrent.TimeUnit

import org.apache.pekko.actor.{Actor, ActorLogging, PoisonPill, Props, Terminated}
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.support.DefaultTransactionDefinition

import scala.concurrent.duration.Duration

object StartMoving
object StartNextBatch

object ArtifactsMoverSupervisor {
  def props(movers: List[ArtifactMover], transactionManager: PlatformTransactionManager): Props =
    Props(new ArtifactsMoverSupervisor(movers, transactionManager))
}

class ArtifactsMoverSupervisor(private val movers: List[ArtifactMover], private val transactionManager: PlatformTransactionManager)
  extends Actor with ActorLogging {

  private val moversItr = movers.iterator

  override def receive: Receive = {
    case StartMoving =>
      log.info(s"Found artifacts to move: ${movers.mkString(", ")}.")
      createAndStartNextActor()
    case Terminated(_) =>
      if (moversItr.hasNext) {
        createAndStartNextActor()
      } else {
        log.info("No more artifacts to move, done.")
      }
      self ! PoisonPill
  }

  private def createAndStartNextActor(): Unit = {
    val mover = moversItr.next()
    log.info(s"Start moving $mover.")
    val actorRef = context.actorOf(ArtifactMoverActor.props(mover, transactionManager))
    actorRef ! StartNextBatch
    context.watch(actorRef)
  }
}

class ArtifactMoverActor(private val mover: ArtifactMover, private val transactionManager: PlatformTransactionManager)
  extends Actor with ActorLogging {

  override def receive: Receive = {
    case StartNextBatch =>
      if (mover.source.isEmpty) {
        self ! PoisonPill
      } else {
        for (_ <- 1 to ArtifactMoverActor.BATCH_SIZE) moveOne()
        scheduleNext()
      }
  }

  private def moveOne(): Unit = {
    val transactionStatus = transactionManager.getTransaction(new DefaultTransactionDefinition())
    try {
      mover.moveOne()
    } catch {
      case t: Throwable =>
        transactionStatus.setRollbackOnly()
        log.error(t, "Last artifact artifact could not be moved. The process will continue. The system will attempt to move again after restart.")
    } finally {
      if (transactionStatus.isRollbackOnly) {
        transactionManager.rollback(transactionStatus)
      } else {
        transactionManager.commit(transactionStatus)
      }
    }
  }

  private def scheduleNext(): Unit = {
    context.system.scheduler.scheduleOnce(ArtifactMoverActor.BATCH_TIMEOUT, self, StartNextBatch)(context.system.dispatcher)
  }
}

object ArtifactMoverActor {
  def props(mover: ArtifactMover, transactionManager: PlatformTransactionManager) =
    Props(new ArtifactMoverActor(mover, transactionManager))

  final val BATCH_SIZE = 10
  final val BATCH_TIMEOUT = Duration(10, TimeUnit.SECONDS)
}
