package com.xebialabs.deployit.repository.sql.artifacts.mover

import com.xebialabs.deployit.engine.spi.event.{SpringInstantiatedListener, SystemStartedEvent}
import com.xebialabs.deployit.repository.sql.artifacts._
import com.xebialabs.xlplatform.scheduler.SchedulerActorSystem
import nl.javadude.t2bus.Subscribe
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Component
import org.springframework.transaction.PlatformTransactionManager

@Component
class ArtifactMoverManager(
                            @Autowired val artifactDataRepository: ArtifactDataRepository,
                            @Autowired val artifactRepository: DbArtifactRepository,
                            @Autowired @Qualifier("mainTransactionManager") val transactionManager: PlatformTransactionManager
                          ) extends SpringInstantiatedListener {

  import ArtifactMoverManager._

  @Subscribe
  def receiveSystemStart(event: SystemStartedEvent): Unit = {
    artifactDataRepository match {
      case p:PrimaryWithFallbacksDataArtifactRepository =>
        val movers: List[ArtifactMover] = p.targets.map {
          case (source: DbArtifactDataRepository, target: FileArtifactDataRepository) =>
            createDbToFileArtifactMover(source, target, artifactRepository)
          case (source: FileArtifactDataRepository, target: DbArtifactDataRepository) =>
            createFileToDbArtifactMover(source, target, artifactRepository)
          case (a1, a2) =>
            throw new IllegalStateException(s"Cannot move artifacts from $a1 to $a2.")
        }
        SchedulerActorSystem.actorSystem.actorOf(ArtifactsMoverSupervisor.props(movers, transactionManager), "ArtifactsMover") ! StartMoving
      case _ => //nothing
    }
  }

}

object ArtifactMoverManager {

  def createDbToFileArtifactMover(source: DbArtifactDataRepository, target: FileArtifactDataRepository, artifactRepository: DbArtifactRepository) =
    new DbToFileArtifactMover(new DbArtifactMoverRepository(source, artifactRepository), new FileArtifactMoverRepository(target))

  def createFileToDbArtifactMover(source: FileArtifactDataRepository, target: DbArtifactDataRepository, artifactRepository: DbArtifactRepository) =
    new FileToDbArtifactMover(new FileArtifactMoverRepository(source), new DbArtifactMoverRepository(target, artifactRepository))
}
