package com.xebialabs.deployit.repository.sql.artifacts.mover

import java.io.FileInputStream
import java.util

import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

trait ArtifactMover {
  def source: ArtifactMoverRepository
  val toMove: Int = source.artifactDataRepository.countArtifacts
  def moveOne(): Unit
}

class DbToFileArtifactMover(override val source: DbArtifactMoverRepository, target: FileArtifactMoverRepository) extends ArtifactMover with Logging {
  private var moved: Int = 0
  private val failed: util.Set[String] = new util.HashSet[String]()
  override def moveOne(): Unit = {
    source.getLastChecksum(failed).foreach { checksum =>
      try {
        logger.debug(s"Moving db artifact [$checksum] to database.")
        val tempFile = source.withInputStream(checksum)(target.storeToTemp)
        val path = target.moveTempToChecksumBasedFile(tempFile, checksum)
        val id = target.insert(path)
        target.insertUsages(id, source.getUsages(checksum))
        source.remove(checksum)
        moved += 1
        logger.info(s"Moved db artifact. [$moved + ${failed.size()} failed /$toMove]")
      } catch {
        case t: Throwable =>
          failed.add(checksum)
          throw t
      }
    }
  }

  override def toString: String = s"$toMove artifacts from db to file"
}

class FileToDbArtifactMover(override val source: FileArtifactMoverRepository, target: DbArtifactMoverRepository) extends ArtifactMover with Logging {
  private var moved: Int = 0
  private val failed: util.Set[Number] = new util.HashSet[Number]()
  override def moveOne(): Unit = {
    source.getLastID(failed).foreach { id =>
      try {
        val file = source.getLocation(id)
        logger.debug(s"Moving file artifact [$id, $file] to database.")
        val checksum = file.getName
        val usages = target.getUsages(checksum).asScala
        usages match {
          case list if list.nonEmpty =>
            logger.debug(s"Skip inserting file artifact id [$id] checksum [$checksum] - it is already in the db")
          case _ =>
            val fis = new FileInputStream(file)
            try {
              target.insert(checksum, fis)
            } finally {
              fis.close()
            }
        }
        target.insertUsages(checksum, source.getUsages(id))
        source.remove(id)
        file.delete()
        moved += 1
        logger.info(s"Moved file artifact. [$moved + ${failed.size()} failed /$toMove]")
      } catch {
        case t: Throwable =>
          failed.add(id)
          throw t
      }
    }
  }

  override def toString: String = s"$toMove artifacts from file to db"
}


