package ai.digital.deploy.metrics.repository.sql

import ai.digital.deploy.metrics.config.Constants
import ai.digital.deploy.metrics.model.{DeploymentMetrics, DeploymentPluginMetrics}
import ai.digital.deploy.metrics.repository.DeploymentMetricsRepository
import ai.digital.deploy.metrics.repository.sql.SqlDeploymentMetricsRepository.{createPluginMapper, createRowMapper}
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{OrderBy, Queries, SchemaInfo, SelectFragmentBuilder, Selectable, SqlFunction, UTC_CALENDAR, toDateTime, SqlCondition => cond}
import com.xebialabs.deployit.repository.sql.persisters.EnvironmentMembersSchema
import com.xebialabs.deployit.sql.base.schema.CIS
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks
import org.joda.time.DateTime
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import grizzled.slf4j.Logging
import org.springframework.transaction.annotation.Transactional
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component

import java.sql.ResultSet
import scala.jdk.CollectionConverters.CollectionHasAsScala

@Component("deploymentMetricsRepositorySql")
@Transactional("mainTransactionManager")
class SqlDeploymentMetricsRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                    (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
      extends DeploymentMetricsRepository with DeploymentMetricsQuery with Logging {

  val DEFAULT_ORDERING: Ordering = new Ordering("path:asc")
    override def findDeployments(path: String, plugins: List[String], owner: Option[String], startDate: Option[String],
                                 taskType: Option[String], paging: Paging, order: Ordering): List[DeploymentMetrics] = {
      if (path.nonEmpty) {
        if (path.startsWith("/Applications")) {
          val builder = new SelectFragmentBuilder(GLOBAL_SELECT_DEPLOYMENT_FRAGMENT)
          builder.where(cond.equals(CIS.name, ArchivedDeploymentTasks.main_application))
          addFilters(builder, path, owner, startDate, taskType, paging, order)
        } else if (path.startsWith(Constants.EnvironmentRoot)) {
          val builder = new SelectFragmentBuilder(FIND_DEPLOYMENT_BY_ENV)
          val envPath = path.substring(1)
          builder.where(cond.likeEscaped(SqlFunction.lower(ArchivedDeploymentTasks.environment),
            s"$envPath%".toLowerCase()))
          withUser(builder, owner)
          withDate(builder, startDate)
          withPage(builder, paging)
          withTaskType(builder, taskType)
          jdbcTemplate.query(builder.query, Setter(builder.parameters), deploymentMapper).asScala.toList
        } else if (path.startsWith(Constants.InfrastructureRoot)) {
          val builder = new SelectFragmentBuilder(FIND_DEPLOYMENT_BY_INFRA)
          builder.where(cond.not(cond.likeEscaped(SqlFunction.lower(CIS.path), s"%$path%/%/%".toLowerCase())))
          builder.where(cond.equals(ArchivedDeploymentTasks.environment_internal_id, EnvironmentMembersSchema.environment_id))
          builder.where(cond.equals(CIS.ID, EnvironmentMembersSchema.member_id))
          addFilters(builder, path, owner, startDate, taskType, paging, order)
        } else {
          throw new IllegalArgumentException("Path doesn't match any of Configuration Items Path")
        }
      } else {
         throw new IllegalArgumentException("Path cannot be empty. Path is required field")
      }
  }
  override def getDeploymentCount(): Int =
    jdbcTemplate.queryForObject(DEPLOYMENT_COUNT, classOf[Number]).intValue()

  override def getUsersCount(owner: String): Int =
    jdbcTemplate.queryForObject(DEPLOYMENT_COUNT_BY_USER, classOf[Number], owner).intValue()

  override def getPluginsData(path: String, plugins: List[String], owner: Option[String], startDate: Option[String],
                     taskType: Option[String], paging: Paging, order: Ordering): List[DeploymentPluginMetrics] = {
    if (path.startsWith(Constants.InfrastructureRoot)) {
      val builder = new SelectFragmentBuilder(FIND_PLUGIN_DATA_BY_INFRA)
      builder.where(cond.equals(ArchivedDeploymentTasks.environment_internal_id, EnvironmentMembersSchema.environment_id))
      builder.where(cond.equals(CIS.ID, EnvironmentMembersSchema.member_id))
      withPluginList(builder, plugins)
      addFilters(builder, path, owner, startDate, taskType, paging, order)
      jdbcTemplate.query(builder.query, Setter(builder.parameters), deploymentPluginMapper).asScala.toList
    } else {
      // When path is either appln , envir
      val builder = new SelectFragmentBuilder(GLOBAL_SELECT_PLUGIN_FRAGMENT)
      builder.where(cond.likeEscaped(SqlFunction.lower(CIS.path.tableAlias("packages")),
        s"%${Constants.ApplicationRoot}%".toLowerCase()))
      builder.where(cond.equals(CIS.parent_id.tableAlias("artifacts"),
        CIS.ID.tableAlias("packages")))
      builder.where(cond.equals(CIS.parent_id.tableAlias("packages"), CIS.ID.tableAlias("appln")))
      builder.where(cond.equals(CIS.name.tableAlias("appln"),
        ArchivedDeploymentTasks.main_application.tableAlias("archivedTask")))
      withPluginList(builder, plugins)
      withUser(builder, owner)
      withDate(builder, startDate)
      withTaskType(builder, taskType)
      jdbcTemplate.query(builder.query, Setter(builder.parameters), deploymentPluginMapper).asScala.toList
    }
  }

  def getPluginsList(): List[DeploymentPluginMetrics] = {
    // No filter
    val builder = new SelectFragmentBuilder(GET_PLUGINS_LIST)
    jdbcTemplate.query(builder.query, deploymentPluginMapper).asScala.toList
  }

  def getPluginsCount(): Int =
    jdbcTemplate.queryForObject(GET_PLUGIN_COUNT, classOf[Number]).intValue()

  private def addFilters(builder: SelectFragmentBuilder, path: String, owner: Option[String],
        startDate: Option[String], taskType: Option[String], paging: Paging, order: Ordering): List[DeploymentMetrics] = {
    withPathPattern(builder, path)
    withUser(builder, owner)
    withDate(builder, startDate)
    withPage(builder, paging)
    withOrder(builder, order)
    withTaskType(builder, taskType)
    debug(builder.query)
    jdbcTemplate.query(builder.query, Setter(builder.parameters), deploymentMapper).asScala.toList
  }

  private def withPathPattern(builder: SelectFragmentBuilder, pathPattern: String) = pathPattern match {
    case pattern if pattern.nonEmpty =>
        builder.where(cond.likeEscaped(SqlFunction.lower(CIS.path), s"%$pattern%".toLowerCase()))
    case _ =>
  }

  private def withPluginList(builder: SelectFragmentBuilder, plugins: List[String]): Any = {
    if (plugins != null && plugins.size > 0) {
      builder.where(cond.in(CIS.ci_type.tableAlias("artifacts"), plugins))
    }
  }

  private def withUser(builder: SelectFragmentBuilder, owner: Option[String]) = owner match {
    case Some(owner) if owner.nonEmpty =>
      builder.where(cond.equals(ArchivedDeploymentTasks.owner, owner))
    case _ =>
  }

  private def withDate(builder: SelectFragmentBuilder, startDate: Option[String]) = startDate match {
    case Some(startDate) if startDate.nonEmpty =>
      val createdDate = DateTime.parse(startDate)
      builder.where(cond.after(ArchivedDeploymentTasks.start_date, createdDate))
    case _ =>
  }

  private def withTaskType(builder: SelectFragmentBuilder, taskType: Option[String]): Any = taskType match {
    case Some(taskType) if taskType.nonEmpty =>
      builder.where(cond.equals(SqlFunction.lower(ArchivedDeploymentTasks.task_type), taskType.toLowerCase()))
    case _ =>
  }

  private def withPage(builder: SelectFragmentBuilder, paging: Paging): Unit =
    Option(paging).filter(_.resultsPerPage != -1).foreach(p => builder.showPage(p.page, p.resultsPerPage))

  private def withOrder(builder: SelectFragmentBuilder, ordering: Ordering): SelectFragmentBuilder = {
    val order = if (ordering == null) DEFAULT_ORDERING else ordering
    val columnToSort = SqlDeploymentMetricsRepository.fieldMapping(order.field)
    builder.orderBy(
      if (order.isAscending) OrderBy.asc(columnToSort) else OrderBy.desc(columnToSort)
    )
  }

  private def deploymentMapper: RowMapper[DeploymentMetrics] = createRowMapper()
  private def deploymentPluginMapper: RowMapper[DeploymentPluginMetrics] = createPluginMapper()
}

object SqlDeploymentMetricsRepository {
  def createRowMapper()(implicit schemaInfo: SchemaInfo): RowMapper[DeploymentMetrics] = (rs: ResultSet, _) =>
                                                          DeploymentMetrics(
    taskId = rs.getString(ArchivedDeploymentTasks.task_id.name),
    ciType = rs.getString(CIS.ci_type.name),
    path = rs.getString(CIS.path.name),
    plugins = List.empty,
    taskType = rs.getString(ArchivedDeploymentTasks.task_type.name),
    owner = rs.getString(ArchivedDeploymentTasks.owner.name),
    // modifiedBy = rs.getString(CIS.modified_by.name),
    startDate =  Option(rs.getTimestamp(ArchivedDeploymentTasks.start_date.name, UTC_CALENDAR)).map(toDateTime).orNull,
    status = rs.getString(ArchivedDeploymentTasks.status.name),
    // modifiedAt = Option(rs.getTimestamp(CIS.modified_at.name, UTC_CALENDAR)).map(toDateTime).orNull
  )

  def createPluginMapper()(implicit schemaInfo: SchemaInfo): RowMapper[DeploymentPluginMetrics] = (rs: ResultSet, _) =>
                                                            DeploymentPluginMetrics(
         taskId = rs.getString(ArchivedDeploymentTasks.task_id.name),
         ciType = rs.getString(CIS.ci_type.name)
   )

  val PATH = "path"
  val fieldMapping: Map[String, Selectable] = Map(
     PATH -> CIS.path.tableAlias("ciTable")
  )
}

trait DeploymentMetricsQuery extends Queries {

  lazy val GLOBAL_SELECT_DEPLOYMENT_FRAGMENT = {
    sqlb"""ciTable.${CIS.ci_type}, ciTable.${CIS.path}, xadtTable.${ArchivedDeploymentTasks.task_id},
          |xadtTable.${ArchivedDeploymentTasks.owner},
          |xadtTable.${ArchivedDeploymentTasks.task_type}, xadtTable.${ArchivedDeploymentTasks.start_date},
          |xadtTable.${ArchivedDeploymentTasks.status}
          |from ${ArchivedDeploymentTasks.tableName} xadtTable, ${CIS.tableName} ciTable"""
  }

  lazy val FIND_DEPLOYMENT_BY_ENV: String = {
    sqlb""" xadtTable.${ArchivedDeploymentTasks.environment} AS ${CIS.path}, 'udm.Environment' AS ${CIS.ci_type},
          |xadtTable.${ArchivedDeploymentTasks.task_id},
          |xadtTable.${ArchivedDeploymentTasks.owner},
          |xadtTable.${ArchivedDeploymentTasks.task_type}, xadtTable.${ArchivedDeploymentTasks.start_date},
          |xadtTable.${ArchivedDeploymentTasks.status}
          |from ${ArchivedDeploymentTasks.tableName} xadtTable"""
  }
  lazy val FIND_DEPLOYMENT_BY_INFRA: String = {
    sqlb"""$GLOBAL_SELECT_DEPLOYMENT_FRAGMENT , ${EnvironmentMembersSchema.tableName} xem """
  }
  val DEPLOYMENT_COUNT = sqlb"select count(*) from  ${ArchivedDeploymentTasks.tableName}"
  val DEPLOYMENT_COUNT_BY_USER = sqlb"select count(*) from ${ArchivedDeploymentTasks.tableName} " +
    sqlb"where ${ArchivedDeploymentTasks.owner} = ?"

  val GLOBAL_SELECT_PLUGIN_FRAGMENT = {
    sqlb"""distinct archivedTask.${ArchivedDeploymentTasks.task_id} , artifacts.${CIS.ci_type}
          |from ${ArchivedDeploymentTasks.tableName} archivedTask, ${CIS.tableName} artifacts,
          |${CIS.tableName} packages, ${CIS.tableName} appln"""
  }

  val FIND_PLUGIN_DATA_BY_INFRA: String = {
    sqlb"""distinct archivedTask.${ArchivedDeploymentTasks.task_id} , artifacts.${CIS.ci_type}
          |from ${ArchivedDeploymentTasks.tableName} archivedTask, ${CIS.tableName} containers,
          |${EnvironmentMembersSchema.tableName} infraMapping"""
  }

  lazy val GET_PLUGINS_LIST: String = {
    sqlb"""distinct containers.${CIS.ci_type} as containers
       from ${CIS.tableName} containers,
            ${ArchivedDeploymentTasks.tableName} archivedTasks, ${EnvironmentMembersSchema.tableName} infraMapping
       where archivedTask.${ArchivedDeploymentTasks.environment_internal_id} =
             infraMapping.${EnvironmentMembersSchema.environment_id} and
       containers.${CIS.ID} = infraMapping.${EnvironmentMembersSchema.member_id} and
       containers.${CIS.path} like '/Infrastructure%'"""
  }

  val GET_PLUGIN_COUNT: String = {
    sqlb"""select count(distinct containers.${CIS.ci_type})
           from ${CIS.tableName} containers,  ${ArchivedDeploymentTasks.tableName} archivedTasks, ${EnvironmentMembersSchema.tableName} infraMapping
       where archivedTasks.${ArchivedDeploymentTasks.environment_internal_id} =
                                 infraMapping.${EnvironmentMembersSchema.environment_id} and
             containers.${CIS.ID} = infraMapping.${EnvironmentMembersSchema.member_id} and
             containers.${CIS.path} like '/Infrastructure%'"""
  }
}

