package ai.digital.deploy.metrics.rest

import ai.digital.deploy.metrics.api.{CiMetricsService, DeploymentMetricsService, MetricsReportService}
import ai.digital.deploy.metrics.config.{Constants, InternalReport}
import ai.digital.deploy.metrics.config.Constants.pageSize
import ai.digital.deploy.metrics.json._
import ai.digital.deploy.metrics.model._
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import org.apache.commons.lang3.exception.ExceptionUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.{BufferedOutputStream, OutputStream}
import java.nio.charset.StandardCharsets
import java.util.zip.{ZipEntry, ZipOutputStream}
import scala.jdk.CollectionConverters._

@Component
class MetricsReportServiceImpl(@Autowired deploymentMetricsService: DeploymentMetricsService,
                               @Autowired listDeploymentJsonWriter: ListDeploymentJsonWriter,
                               @Autowired listConfigurationItemsJsonWriter: ListConfigurationItemsJsonWriter,
                               @Autowired configurationItemsCountJsonWriter: ConfigurationItemsCountJsonWriter,
                               @Autowired usersCountByCiJsonWriter: UsersCountByCiJsonWriter,
                               @Autowired pluginsCountByCiJsonWriter: PluginsCountByCiJsonWriter,
                               @Autowired subFoldersCountByCiJsonWriter: SubFoldersCountByCiJsonWriter,
                               @Autowired deploymentsCountJsonWriter: DeploymentsCountJsonWriter,
                               @Autowired pluginsCountByDeploymentJsonWriter: PluginsCountByDeploymentJsonWriter,
                               @Autowired ciMetricsService: CiMetricsService) extends MetricsReportService {

  override def prepareMetricsReportZip(output: OutputStream,
                                       reportsToZip: List[InternalReport]): Unit =
    using(new ZipOutputStream(asBuffered(output))) { zipOutputStream => {
      reportsToZip.asJava.forEach(reportPath => reportToZip(zipOutputStream, reportPath))
      zipOutputStream.flush()
    }
    }

  private def asBuffered(os: OutputStream): OutputStream =
    if (os.isInstanceOf[BufferedOutputStream]) os
    else new BufferedOutputStream(os)

  private def reportToZip(zipOutputStream: ZipOutputStream, internalReport: InternalReport): Unit = {
    try {
      val reportData = internalReport.reportSupplier()
      withNewZipEntry(s"${internalReport.reportName}.${internalReport.fileType}", zipOutputStream,
        () => zipOutputStream.write(reportData.getBytes(StandardCharsets.UTF_8)))
    } catch {
      case e: Exception =>
        val exceptionData = s"${e.getMessage}${System.lineSeparator}${ExceptionUtils.getStackTrace(e)}"
        withNewZipEntry(s"${internalReport.reportName}.err", zipOutputStream,
          () => zipOutputStream.write(exceptionData.getBytes(StandardCharsets.UTF_8)))
    }
  }

  private def withNewZipEntry(zipEntryName: String, zipOutputStream: ZipOutputStream, writer: () => Unit): Unit = {
    zipOutputStream.putNextEntry(new ZipEntry(zipEntryName))
    writer()
    zipOutputStream.closeEntry()
  }

  override def listDeployments: String = {
    var lines: List[DeploymentMetrics] = List.empty
    Constants.paths.foreach(path =>
      lines :::=
        deploymentMetricsService.listDeployments(DeploymentMetricsFilter(path, null, "admin",
          Constants.creationDate, ""), pageSize, null)
    )
    listDeploymentJsonWriter.toJson(lines.asJava)
  }

  override def getDeploymentsCount: String = {
    deploymentsCountJsonWriter.toJson(deploymentMetricsService.getDeploymentsCount())
  }

  override def getPluginsCountByDeployment: String = {
    pluginsCountByDeploymentJsonWriter.toJson(deploymentMetricsService.getPluginsCountByDeployment())
  }

  override def listConfigurationItems: String = {
    var lines: List[CiPluginMetrics] = List.empty
    Constants.paths.foreach(path =>
      lines :::=
        ciMetricsService.listConfigurationItems(CiMetricsFilter(path, null, "admin",
          Constants.creationDate), pageSize, null)
    )
    listConfigurationItemsJsonWriter.toJson(lines.asJava)
  }

  override def getConfigurationItemsCount: String = {
    var cis: List[CiCount] = List.empty
    Constants.paths.foreach(path =>
      cis = cis.appended(
        ciMetricsService.getConfigurationItemsCount(CiMetricsFilter(path, null, "admin",
          Constants.creationDate))
      ))
    configurationItemsCountJsonWriter.toJson(cis.asJava)
  }

  override def getUsersCountByCI: String = {
    val usersCountByCi: UsersCountByCi = ciMetricsService.getUsersCountByCI
    usersCountByCiJsonWriter.toJson(usersCountByCi)
  }

  override def getPluginsCountByCI: String = {
    val pluginsCountByCi: PluginsCountByCi = ciMetricsService.getPluginsCountByCI
    pluginsCountByCiJsonWriter.toJson(pluginsCountByCi)
  }

  override def getSubFoldersCountByCI: String = {
    var foldersCount: List[FoldersCount] = List.empty
    Constants.paths.foreach(path =>
      foldersCount = foldersCount.appended(
        ciMetricsService.getSubFoldersCountByCI(path))
    )
    subFoldersCountByCiJsonWriter.toJson(foldersCount.asJava)
  }

}
