package com.xebialabs.xlplatform.extensions.maven

import java.io.{ByteArrayInputStream, FileInputStream, InputStream}
import java.net.URI
import java.util

import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver
import com.xebialabs.deployit.engine.spi.artifact.resolution.{ArtifactResolver, CannotLocateArtifactException, ResolvedArtifactFile}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.proxy.ProxyServer
import com.xebialabs.deployit.repository.WorkDir
import grizzled.slf4j.Logging
import org.apache.maven.repository.internal.MavenRepositorySystemUtils
import org.eclipse.aether.artifact.{Artifact, DefaultArtifact}
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory
import org.eclipse.aether.repository.{Authentication, LocalRepository}
import org.eclipse.aether.resolution.{ArtifactRequest, ArtifactResolutionException}
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory
import org.eclipse.aether.spi.connector.transport.TransporterFactory
import org.eclipse.aether.transport.file.FileTransporterFactory
import org.eclipse.aether.transport.http.HttpTransporterFactory
import org.eclipse.aether.util.repository.DefaultProxySelector
import org.eclipse.aether.{RepositorySystem, repository}

import scala.collection.convert.ImplicitConversions._
import scala.util.{Failure, Success, Try}

@Resolver(protocols = Array("maven"))
class MavenArtifactResolver extends ArtifactResolver with Logging {

  private[maven] lazy val mavenSettings = MavenSettings()

  private[maven] lazy val system: RepositorySystem = configureRepositorySystem()

  private[maven] def getFileUri(artifact: SourceArtifact) = URI.create(artifact.getFileUri)

  private[maven] def getWorkDirFactory = mavenSettings.workDirFactory

  private[maven] def getRepositories = mavenSettings.repositories

  override def resolveLocation(artifact: SourceArtifact): ResolvedArtifactFile = {

    val session = MavenRepositorySystemUtils.newSession()

    Option(artifact.getProxySettings).map(_.asInstanceOf[ProxyServer]).foreach(proxyServer => {
      val proxySelector: DefaultProxySelector = new DefaultProxySelector
      var proxy: repository.Proxy = null
      if (proxyServer.getUsername != null) {
        proxy = new repository.Proxy(
          proxyServer.getProtocol.getValue, proxyServer.getHostname, proxyServer.getPort, newAuthentication(proxyServer.getUsername, proxyServer.getPassword))
      }
      else {
        proxy = new repository.Proxy(proxyServer.getProtocol.getValue, proxyServer.getHostname, proxyServer.getPort)
      }
      proxySelector.add(proxy, "localhost")
      session.setProxySelector(proxySelector)
    })
//


    val workDir = getWorkDirFactory.newWorkDir("maven")
    val uri: URI = getFileUri(artifact)
    val request = createRequest(uri.getSchemeSpecificPart)
    try {

      val localRepo = new LocalRepository(workDir.getPath)
      session.setLocalRepositoryManager(system.newLocalRepositoryManager(session, localRepo))

      val atf = system.resolveArtifact(session, request).getArtifact
      new ResolvedArtifactFile {
        override def getFileName: String = atf.getFile.getName

        override def openStream: InputStream = new FileInputStream(atf.getFile)

        override def close(): Unit = workDir.delete()
      }

    } catch {
      case e: Exception =>
        handleError(request, workDir, e)
    }
  }

  import org.eclipse.aether.repository.{AuthenticationContext, AuthenticationDigest}

  private def newAuthentication(username: String, password: String) = new Authentication {
    override def fill(context: AuthenticationContext, key: String, data: util.Map[String, String]): Unit = {
      context.put(AuthenticationContext.USERNAME, username)
      context.put(AuthenticationContext.PASSWORD, password)
    }

    override def digest(digest: AuthenticationDigest): Unit = {
      digest.update(AuthenticationContext.USERNAME, username, AuthenticationContext.PASSWORD, password)
    }
  }

  private def handleError(artifactRequest: ArtifactRequest, workDir: WorkDir, e: Throwable) = e match {
    case ex: ArtifactResolutionException if ex.getResult.isMissing && mavenSettings.ignoreMissingArtifact =>
      logger.info(s"ignoreMissingArtifact mode: Could not find artifact ${artifactRequest.getArtifact} in the maven repositories, creating an empty fallback artifact.")
      createDummyArtifact(artifactRequest, workDir)
    case ex: CannotLocateArtifactException =>
      workDir.delete()
      throw ex
    case _ =>
      workDir.delete()
      throw new CannotLocateArtifactException(e)
  }

  private def createDummyArtifact(artifactRequest: ArtifactRequest, workDir: WorkDir): ResolvedArtifactFile = {
    new ResolvedArtifactFile {
      override def openStream(): InputStream = new ByteArrayInputStream(s"Could not locate ${artifactRequest.getArtifact} in the maven repository.".getBytes("UTF-8"))

      override def getFileName: String = {
        val a: Artifact = artifactRequest.getArtifact
        if (a.getClassifier.isEmpty) {
          s"${a.getArtifactId}-${a.getVersion}.${a.getExtension}"
        } else {
          s"${a.getArtifactId}-${a.getVersion}-${a.getClassifier}.${a.getExtension}"
        }
      }

      override def close(): Unit = workDir.delete()
    }
  }

  override def validateCorrectness(artifact: SourceArtifact): Boolean = getFileUri(artifact).getScheme.equals("maven") &&
    (Try(new DefaultArtifact(getFileUri(artifact).getSchemeSpecificPart)) match {
      case Success(_) => true
      case Failure(e: IllegalArgumentException) => false
      case Failure(e@_) => throw e
    })

  private def configureRepositorySystem(): RepositorySystem = {
    val locator = MavenRepositorySystemUtils.newServiceLocator()
      .addService(classOf[RepositoryConnectorFactory], classOf[BasicRepositoryConnectorFactory])
      .addService(classOf[TransporterFactory], classOf[FileTransporterFactory])
      .addService(classOf[TransporterFactory], classOf[HttpTransporterFactory])

    locator.getService(classOf[RepositorySystem])
  }

  private def createRequest(artifactCoordinates: String): ArtifactRequest = {
    val request = new ArtifactRequest()
    request.setArtifact(new DefaultArtifact(artifactCoordinates))
    request.setRepositories(getRepositories)
  }
}
