package com.xebialabs.license.service.aws

import com.xebialabs.license.service.LicenseConfig
import grizzled.slf4j.Logging
import org.apache.http.client.config.RequestConfig
import org.apache.http.client.methods.HttpGet
import org.bouncycastle.asn1.cms.ContentInfo
import org.bouncycastle.cert.X509CertificateHolder
import org.bouncycastle.cms.CMSSignedData
import org.bouncycastle.cms.jcajce.JcaSignerInfoVerifierBuilder
import org.bouncycastle.openssl.PEMParser
import org.bouncycastle.operator.bc.BcDigestCalculatorProvider
import spray.json._

import java.io.StringReader
import scala.jdk.CollectionConverters._

trait AWSAmiMarketplace extends Logging with HttpClientAware {
  val licenseConfig: LicenseConfig

  def getProductCodes: List[String] = {
    val data: CMSSignedData = getSignedMetadata
    validateMetadataSignature(data)

    val content = new String(data.getSignedContent.getContent.asInstanceOf[Array[Byte]])
    val document = content.parseJson
    document.asJsObject.fields("marketplaceProductCodes") match {
      case JsNull => List(licenseConfig.license.ami.productCode)
      case JsString(s) => List(s)
      case JsArray(l) => l.collect { case JsString(s) => s }.toList
      case w@_ => throw new Exception(s"Unexpected Json type: $w")
    }
  }

  private[aws] def validateMetadataSignature(data: CMSSignedData): Unit = {
    val certificate = new PEMParser(new StringReader(licenseConfig.license.ami.awsCertificate)).readObject().asInstanceOf[X509CertificateHolder]
    val infos = data.getSignerInfos
    infos.getSigners.asScala.find { si =>
      val verifier = new JcaSignerInfoVerifierBuilder(new BcDigestCalculatorProvider).build(certificate)
      si.verify(verifier)
    } match {
      case Some(si) => // OK
        info("Succesfully verified AWS Signature, running in EC2")
      case None => throw AWSLicenseException("Could not verify AWS signature on identity-document")
    }
  }

  private[aws] def getSignedMetadata = {
    val documentSignature = getMetadata(licenseConfig.license.ami.identityDocumentSignature)
    val signature =
      s"""
         |-----BEGIN PKCS7-----
         |$documentSignature
         |-----END PKCS7-----
      """.stripMargin
    val obj: ContentInfo = new PEMParser(new StringReader(signature)).readObject().asInstanceOf[ContentInfo]
    new CMSSignedData(obj)
  }

  private[aws] def getMetadata(locator: String) = {
    val amiConfig = licenseConfig.license.ami
    val identityDocumentEndpoint = amiConfig.metadataEndpoint + locator
    getHttpClient().execute(new HttpGet(identityDocumentEndpoint), responseHandler()) match {
      case Some(x) => x
      case None => throw AWSLicenseException("Could not retrieve the identity-document from the AMI")
    }
  }


  override private[aws] def requestConfig = {
    val amiConfig = licenseConfig.license.ami
    RequestConfig.custom.setConnectTimeout(amiConfig.connectTimeout).build
  }
}
