package com.xebialabs.license.service;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import org.joda.time.LocalDate;
import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.RepositoryMetadataService;
import com.xebialabs.license.*;

import static java.lang.String.format;

public abstract class AbstractLicenseService implements LicenseService {

    protected static final Logger log = LoggerFactory.getLogger(AbstractLicenseService.class);
    protected static final int MINIMUM_LICENSE_VERSION = 3;

    protected final Clock clock;

    protected final File licenseFile;
    protected final String product;
    protected final Set<String> allowedEditions = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(LicenseVersion3.Edition.Trial.name(), LicenseVersion3.Edition.Enterprise.name())));

    protected License license;
    protected LicenseCiCounter ciCounter;
    protected LicenseCiCounterFactory ciCounterFactory = new DefaultLicenseCiCounterFactory();
    protected RepositoryMetadataService repositoryMetadataService;

    public AbstractLicenseService(String licensePath, String product, RepositoryMetadataService repositoryMetadataService) throws IOException {
        this(licensePath, product, repositoryMetadataService, new RealClock());
    }

    public AbstractLicenseService(String licensePath, String product, RepositoryMetadataService repositoryMetadataService, Clock clock) {
        this.product = product;
        this.licenseFile = new File(licensePath);
        this.repositoryMetadataService = Checks.checkNotNull(repositoryMetadataService, "repositoryMetadataService");
        this.clock = clock;
    }

    @Override
    public void initialize(LicenseCiCounterFactory ciCounterFactory) {
        this.ciCounterFactory = ciCounterFactory;
        reLoadCiCounter();
    }

    protected void logLicenseError(File licenseFile) {
        log.warn("*** Please contact your XebiaLabs sales representative to obtain a valid license.");
        log.warn("*** Place a valid license in {} to continue using the product.", licenseFile.getAbsolutePath());
    }

    protected void logLicenseVersionError(License license, int minimumLicenseVersion) {
        log.error("*** Your license is version {}, but a minimum version of {} is required", license.getLicenseVersion(), minimumLicenseVersion);
        log.error("*** Please contact your XebiaLabs sales representative to obtain a valid license.");
    }

    protected void reLoadCiCounter() {
        log.debug("Loading license CI counter using current license.");
        ciCounter = ciCounterFactory.create(license);
        log.debug("License CI counter loaded and contains {} allowed CIs.", ciCounter.allowedCiAmounts().size());
    }

    protected void logLicenseEditionError(final License license) {
        log.error("*** Your license has edition {}, but only {} is allowed", license.getStringValue(LicenseProperty.EDITION), allowedEditions);
        log.error("*** Please contact your XebiaLabs sales representative to obtain a valid license.");
    }

    @Override
    public String getProduct() {
        return product;
    }

    @Override
    public LicenseCiCounter getCounter() {
        if (ciCounter == null) {
            throw new IllegalStateException("ciCounter is null. Did you forget to initialize the license server?");
        }
        return ciCounter;
    }

    @Override
    public boolean isLicenseExpired() {
        return license.isDateExpired(new LocalDate(clock.millis()));
    }

    @Override
    public Period getValidPeriod() {
        final LocalDate localDateValue = license.getLocalDateValue(LicenseProperty.EXPIRES_AFTER);
        final LocalDate now = new LocalDate(clock.millis());

        if (now.isAfter(localDateValue)) {
            return Period.ZERO;
        }
        return new Period(now, localDateValue, PeriodType.days());
    }

    @Override
    public void validate() throws LicenseViolationException {
        validateLicenseProperties(license);
        ciCounter.validate();
    }

    @Override
    public void validate(License license) throws LicenseViolationException {
        validateLicenseProperties(license);

        LicenseCiCounter tmpCiCounter = ciCounterFactory.create(license);
        tmpCiCounter.validate();
    }

    @Override
    public void reload() throws LicenseViolationException, LicenseParseException {
        license = readLicense(licenseFile);
        reLoadCiCounter();
        validate();
    }

    @Override
    public License getLicense() {
        return license;
    }

    @Override
    public LicenseTransaction startTransaction() {
        return new LicenseTransaction();
    }

    @Override
    public void rollbackTransaction(final LicenseTransaction transaction) {
        for (Type type : transaction.getCiCountTypes()) {
            ciCounter.getAtomicCiCount(type).addAndGet(-transaction.getCiCount(type));
        }
    }

    protected void validateLicenseProperties(final License license) throws LicenseViolationException {
        if (license.isDummyLicense()) {
            return;
        }
        String licenseProduct = license.getStringValue(LicenseProperty.PRODUCT);
        if (!this.product.equals(licenseProduct)) {
            throw new LicenseProductException(format("license is issued for %s, but used in %s", licenseProduct, product));
        }

        if (license.isDateExpired(new LocalDate(clock.millis()))) {
            throw new LicensePeriodExpiredException("the license has expired");
        }

        if (!license.isAtLeastVersion(MINIMUM_LICENSE_VERSION)) {
            throw new LicenseVersionException(format("for this version of %s, a license version %d or higher is required", license.getStringValue(LicenseProperty.PRODUCT), MINIMUM_LICENSE_VERSION));
        }

        if (!isValidEdition(license)) {
            throw new LicenseEditionException(format("for this version of %s, the license must be one of %s", license.getStringValue(LicenseProperty.PRODUCT), allowedEditions));
        }

        final String stringValue = license.getStringValue(LicenseProperty.REPOSITORY_ID);
        if (Strings.isNotEmpty(stringValue)) {
            // Note, the following invocation can throws a com.xebialabs.license.LicenseRepositoryIdException.
            repositoryMetadataService.validateAndStoreRepositoryId(stringValue);
        }
    }

    protected boolean isValidEdition(License license) {
        return allowedEditions.contains(license.getStringValue(LicenseProperty.EDITION));
    }

    protected abstract License readLicense(File licenseFile) throws LicenseParseException, LicenseViolationException;
}
