package com.xebialabs.license.service;

import com.xebialabs.license.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;

import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;

public class LicenseInstallServiceImpl implements LicenseInstallService {

    private final File licenseFile;
    private final LicenseService licenseService;

    private static final Logger logger = LoggerFactory.getLogger(LicenseInstallServiceImpl.class);

    public LicenseInstallServiceImpl(String licensePath, final LicenseService licenseService) {
        this.licenseFile = new File(licensePath);
        this.licenseService = licenseService;
    }

    /**
     * Best effort attempt to check if we can actually write the license.
     */
    boolean canInstallNewLicense() {
        if (!licenseFile.exists()) {
            return licenseFile.getParentFile().canWrite();
        }
        return licenseFile.canWrite();
    }

    @Override
    public License installNewLicense(final String licenseText) throws LicenseInstallationFailedException {
        if (!canInstallNewLicense()) {
            throw new LicenseInstallationFailedException("Could not install license because the license file could not be written to the file system.");
        }
        try {
            License tmpLicense = readLicense(licenseText);

            licenseService.validate(tmpLicense); // validates expired and repository, throws exception when invalid

            String plainText = LicenseUtils.decodeIfNecessary(licenseText);

            Files.write(licenseFile.toPath(), plainText.getBytes(UTF_8));

            licenseService.reload();

            return tmpLicense;
        } catch (LicenseRepositoryIdException e) {
            throw new LicenseInstallationFailedException("Could not install license because the repository is locked to another license", e);
        } catch (LicenseViolationException e) {
            throw new LicenseInstallationFailedException("Could not install license because it is invalid", e);
        } catch (IOException e) {
            throw new LicenseInstallationFailedException("Could not install license due to problems with the file system", e);
        }
    }

    @Override
    public String getLicenseRenewalMessage() {
        if (!licenseFile.exists()) {
            return null;
        }

        try {
            licenseService.validate();
        } catch (LicenseVersionException e) {
            logger.info("License status: ", e.getMessage());
            return "Your license is of an old version.";
        } catch (LicensePeriodExpiredException e) {
            logger.info("License status: ", e.getMessage());
            return "Your license has expired.";
        } catch (LicenseEditionException e) {
            logger.info("License status: ", e.getMessage());
            License license = licenseService.getLicense();
            if (license != null && "Community".equals(license.getStringValue(LicenseProperty.EDITION))) {
                return format("This edition of %s cannot be used with a Community Edition license.", licenseService.getProduct());
            }
            return format("This edition of %s requires a Trial Edition license or an Enterprise Edition license.", licenseService.getProduct());
        } catch (AmountOfCisExceededException e) {
            logger.info("License status: ", e.getMessage());
            return "The number of configuration items in your repository exceeds the limit on this license.";
        } catch (LicenseViolationException e) {
            logger.info("License status: ", e.getMessage());
            return format("The current license is not valid (%s).", e.getMessage());
        }
        return "Enter a new license key to renew your license.";
    }

    private License readLicense(String licenseText) throws IOException {
        File tempFile = File.createTempFile("license", "tmp");
        Files.write(tempFile.toPath(), licenseText.getBytes(UTF_8));
        LicenseReader licenseReader = new LicenseReader();
        return licenseReader.readLicense(tempFile);
    }
}
