package com.xebialabs.license.service;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.repository.RepositoryMetadataService;
import com.xebialabs.license.*;

public class LicenseServiceImpl extends AbstractLicenseService {

    private static final Logger log = LoggerFactory.getLogger(LicenseServiceImpl.class);

    private final LicenseReader licenseReader = new LicenseReader();

    public LicenseServiceImpl(String licensePath, String product, RepositoryMetadataService repositoryMetadataService) throws IOException,
            LicenseParseException, LicenseViolationException {
        super(licensePath, product, repositoryMetadataService);

        if (!licenseFile.exists()) {
            logLicenseError(licenseFile);
            throw new FileNotFoundException("License cannot be found at " + licenseFile.getAbsolutePath());
        }

        license = readLicense(licenseFile);

        if (license.isDateExpired()) {
            logLicenseError(licenseFile);
        }

        if (!license.isAtLeastVersion(MINIMUM_LICENSE_VERSION)) {
            logLicenseVersionError(license, MINIMUM_LICENSE_VERSION);
        }

        if (!isValidEdition(license)) {
            logLicenseEditionError(license);
        }
    }

    /**
     * Constructor with injected clock for testing. Protected to make it only available for tests in the same package.
     */
    protected LicenseServiceImpl(String licensePath, String product, RepositoryMetadataService repositoryMetadataService, Clock clock) throws IOException,
            LicenseParseException, LicenseViolationException {
        super(licensePath, product, repositoryMetadataService, clock);

        if (!licenseFile.exists()) {
            logLicenseError(licenseFile);
            throw new FileNotFoundException("License cannot be found at " + licenseFile.getAbsolutePath());
        }

        license = readLicense(licenseFile);

        if (license.isDateExpired()) {
            logLicenseError(licenseFile);
        }

        if (!license.isAtLeastVersion(MINIMUM_LICENSE_VERSION)) {
            logLicenseVersionError(license, MINIMUM_LICENSE_VERSION);
        }

        if (!isValidEdition(license)) {
            logLicenseEditionError(license);
        }
    }

    protected License readLicense(File licenseFile) throws LicenseParseException, LicenseViolationException {
        String licenseFilePath = licenseFile.getAbsolutePath();

        log.info("Reading license from file {}", licenseFilePath);
        License license = licenseReader.readLicense(licenseFile);
        log.info(license.toLicenseContent());

        if (license.isDateExpired()) {
            log.warn("*** License has expired.");
        }

        return license;
    }
}
