package com.xebialabs.license.service;

import java.util.*;
import org.joda.time.LocalDate;
import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.io.ConfigurationResource;
import com.xebialabs.deployit.repository.RepositoryMetadataService;
import com.xebialabs.license.*;

import static java.lang.String.format;

public abstract class AbstractResourceLicenseService extends BaseLicenseService implements LicenseService {

    protected static final Logger log = LoggerFactory.getLogger(AbstractResourceLicenseService.class);
    protected final Collection<LicenseType> allowedLicenseTypes = EnumSet.of(LicenseType.MARKETPLACE, LicenseType.VERSION_3, LicenseType.VERSION_4);

    protected final Clock clock;

    protected final ConfigurationResource licenseResource;
    protected final String product;
    protected final Set<String> allowedEditions = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            LicenseVersion4.Edition4.Trial.name(),
            LicenseVersion4.Edition4.Essentials.name(),
            LicenseVersion4.Edition4.Pro.name(),
            LicenseVersion4.Edition4.Premium.name(),
            LicenseVersion4.Edition4.Unregistered.name(),
            LicenseVersion3.Edition3.Enterprise.name()
    )));

    protected License license;
    protected RepositoryMetadataService repositoryMetadataService;

    public AbstractResourceLicenseService(ConfigurationResource licenseResource, String product, RepositoryMetadataService repositoryMetadataService) {
        this(licenseResource, product, repositoryMetadataService, new RealClock());
    }

    public AbstractResourceLicenseService(ConfigurationResource licenseResource, String product, RepositoryMetadataService repositoryMetadataService, Clock clock) {
        this.product = product;
        this.licenseResource = licenseResource;
        this.repositoryMetadataService = Checks.checkNotNull(repositoryMetadataService, "repositoryMetadataService");
        this.clock = clock;
    }

    @Override
    public void initialize(LicenseCiCounterFactory ciCounterFactory) {
        super.initialize(ciCounterFactory, license);
    }

    protected void logLicenseError(ConfigurationResource licenseResource) {
        log.warn("*** Please contact your XebiaLabs sales representative to obtain a valid license.");
        log.warn("*** Place a valid license in {} to continue using the product.", licenseResource.getPath());
    }

    protected void logLicenseVersionError(License license) {
        log.error("*** Your license is type {}, but one of {} is required", license.getLicenseVersion(), allowedLicenseTypes);
        log.error("*** Please contact your XebiaLabs sales representative to obtain a valid license.");
    }

    protected void logLicenseEditionError(final License license) {
        log.error("*** Your license has edition {}, but only {} is allowed", license.getStringValue(LicenseProperty.EDITION), allowedEditions);
        log.error("*** Please contact your XebiaLabs sales representative to obtain a valid license.");
    }

    @Override
    public String getProduct() {
        return product;
    }

    @Override
    public boolean isLicenseExpired() {
        return license.isDateExpired(new LocalDate(clock.millis()));
    }

    @Override
    public Period getValidPeriod() {
        final LocalDate localDateValue = license.getLocalDateValue(LicenseProperty.EXPIRES_AFTER);
        final LocalDate now = new LocalDate(clock.millis());

        if (localDateValue == null) {
            return Period.years(1000);
        }
        if (now.isAfter(localDateValue)) {
            return Period.ZERO;
        }
        return new Period(now, localDateValue, PeriodType.days());
    }

    @Override
    public void validate() throws LicenseViolationException {
        validateLicenseProperties(license);
        super.validate();
    }

    @Override
    public void validate(License license) throws LicenseViolationException {
        validateLicenseProperties(license);
        super.validate(license);
    }

    @Override
    public void reload() throws LicenseViolationException, LicenseParseException {
        license = readLicense(licenseResource);
        reLoadCiCounter(license);
        validate();
    }

    @Override
    public License getLicense() {
        return license;
    }


    protected void validateLicenseProperties(final License license) throws LicenseViolationException {
        if (license.isDummyLicense()) {
            return;
        }
        String licenseProduct = license.getStringValue(LicenseProperty.PRODUCT);
        if (!this.product.equals(licenseProduct)) {
            throw new LicenseProductException(format("license is issued for %s, but used in %s", licenseProduct, product));
        }

        if (license.isDateExpired(new LocalDate(clock.millis()))) {
            if (Arrays.asList(LicenseType.VERSION_3, LicenseType.VERSION_4).contains(LicenseType.fromType(license.getStringValue(LicenseProperty.LICENSE_VERSION))) &&
                    license.getStringValue(LicenseProperty.EDITION).equals(LicenseVersion4.Edition4.Unregistered.name())) {
                throw new UnregisteredLicenseExpiredException("the unregistered version of the license has expired");
            } else {
                throw new LicensePeriodExpiredException("the license has expired");
            }
        }

        if (!isAllowedLicenseType(license)) {
            throw new LicenseVersionException(format("for this version of %s, only license types {} are allowed", license.getStringValue(LicenseProperty.PRODUCT), allowedLicenseTypes));
        }

        if (!isValidEdition(license)) {
            throw new LicenseEditionException(format("for this version of %s, the license must be one of %s", license.getStringValue(LicenseProperty.PRODUCT), allowedEditions));
        }

        final String stringValue = license.getStringValue(LicenseProperty.REPOSITORY_ID);
        if (Strings.isNotEmpty(stringValue)) {
            // Note, the following invocation can throws a com.xebialabs.license.LicenseRepositoryIdException.
            repositoryMetadataService.validateAndStoreRepositoryId(stringValue);
        }
    }

    protected boolean isValidEdition(License license) {
        return allowedEditions.contains(license.getStringValue(LicenseProperty.EDITION));
    }

    protected boolean isAllowedLicenseType(License license) {
        return allowedLicenseTypes.contains(LicenseType.fromType(license.getStringValue(LicenseProperty.LICENSE_VERSION)));
    }

    protected abstract License readLicense(ConfigurationResource licenseResource) throws LicenseParseException, LicenseViolationException;
}
