package com.xebialabs.license.service;

import com.xebialabs.deployit.io.ConfigurationResource;
import com.xebialabs.deployit.repository.RepositoryMetadataService;
import com.xebialabs.license.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileNotFoundException;
import java.io.IOException;

public class ResourceLicenseServiceImpl extends AbstractResourceLicenseService {

    private static final Logger log = LoggerFactory.getLogger(ResourceLicenseServiceImpl.class);

    private final LicenseReader licenseReader = new LicenseReader();

    public ResourceLicenseServiceImpl(ConfigurationResource licenseResource, String product, RepositoryMetadataService repositoryMetadataService) throws IOException,
            LicenseParseException, LicenseViolationException {
        super(licenseResource, product, repositoryMetadataService);

        if (!licenseResource.exists()) {
            logLicenseError(licenseResource);
            throw new FileNotFoundException("License cannot be found at " + licenseResource.getPath());
        }

        license = readLicense(licenseResource);

        if (license.isDateExpired()) {
            logLicenseError(licenseResource);
        }

        if (!isAllowedLicenseType(license)) {
            logLicenseVersionError(license);
        }

        if (!isValidEdition(license)) {
            logLicenseEditionError(license);
        }
    }

    /**
     * Constructor with injected clock for testing. Protected to make it only available for tests in the same package.
     */
    ResourceLicenseServiceImpl(ConfigurationResource licenseResource, String product, RepositoryMetadataService repositoryMetadataService, Clock clock) throws IOException,
            LicenseParseException, LicenseViolationException {
        super(licenseResource, product, repositoryMetadataService, clock);

        if (!licenseResource.exists()) {
            logLicenseError(licenseResource);
            throw new FileNotFoundException("License cannot be found at " + licenseResource.getPath());
        }

        license = readLicense(licenseResource);

        if (license.isDateExpired()) {
            logLicenseError(licenseResource);
        }

        if (!isAllowedLicenseType(license)) {
            logLicenseVersionError(license);
        }

        if (!isValidEdition(license)) {
            logLicenseEditionError(license);
        }
    }

    protected License readLicense(ConfigurationResource licenseResource) throws LicenseParseException, LicenseViolationException {
        try {
            log.info("Reading license from file {}", licenseResource.getPath());
            License license = licenseReader.readLicense(licenseResource);
            log.info(license.toLicenseContent());
            if (license.isDateExpired()) {
                log.warn("*** License has expired.");
            }

            return license;
        } catch (IOException e) {
            throw new LicenseParseException(e.getMessage(), e);
        }
    }
}
