package com.xebialabs.license.service;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.license.*;

import static com.xebialabs.license.LicenseType.VERSION_3;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;

public class LicenseInstallServiceImpl implements LicenseInstallService {

    private final File licenseFile;
    private final LicenseService licenseService;

    private static final Logger logger = LoggerFactory.getLogger(LicenseInstallServiceImpl.class);

    // Client messages
    private static final String LICENSE_STATUS = "License status: ";
    private static final String LICENSE_OLD_VERSION = "Your license is of an old version.";
    private static final String LICENSE_EXPIRED = "Your license has expired.";
    private static final String LICENSE_CI_EXCEEDS_LIMIT = "The number of configuration items in your repository exceeds the limit on this license.";
    private static final String LICENSE_ENTER_NEW = "Enter a new license key to renew your license.";
    private static final String LICENSE_CONTINUE_TRIAL = "Enter your license key to extend your trial.";
    private static final String LICENSE_UNREGISTERED_EXPIRED = "Your unregistered license for %s has expired.";
    private static final String LICENSE_NOT_VALID = "The current license is not valid (%s).";
    private static final String LICENSE_COMMUNITY_MIX = "This edition of %s cannot be used with a Community Edition license.";
    private static final String LICENSE_ENTERPRISE_OR_TRIAL = "This edition of %s requires a Trial Edition license or an Enterprise Edition license.";
    private static final String LICENSE_ESSENTIALS_PRO_PREMIUM_OR_TRIAL = "This edition of %s requires a Trial, an Essentials, a Pro or a Premium Edition license.";
    private static final String COMMUNITY = "Community";

    // Exception messages
    private static final String CANNOT_BE_WRITTEN = "Could not install license because the license file could not be written to the file system.";
    private static final String REPO_IS_LOCKED = "Could not install license because the repository is locked to another license";
    private static final String INVALID = "Could not install license because it is invalid";
    private static final String FS_ERROR = "Could not install license due to problems with the file system";

    public LicenseInstallServiceImpl(String licensePath, final LicenseService licenseService) {
        this.licenseFile = new File(licensePath);
        this.licenseService = licenseService;
    }

    /**
     * Best effort attempt to check if we can actually write the license.
     */
    boolean canInstallNewLicense() {
        if (!licenseFile.exists()) {
            return licenseFile.getParentFile().canWrite();
        }
        return licenseFile.canWrite();
    }

    @Override
    public License installNewLicense(final String licenseText) throws LicenseInstallationFailedException {
        if (!canInstallNewLicense()) {
            logger.info(CANNOT_BE_WRITTEN);
            throw new LicenseInstallationFailedException(CANNOT_BE_WRITTEN);
        }
        try {
            License tmpLicense = readLicense(licenseText);

            licenseService.validate(tmpLicense); // validates expired and repository, throws exception when invalid

            String plainText = LicenseUtils.decodeIfNecessary(licenseText);

            Files.write(licenseFile.toPath(), plainText.getBytes(UTF_8));

            licenseService.reload();

            return tmpLicense;
        } catch (LicenseRepositoryIdException e) {
            logger.info(REPO_IS_LOCKED);
            throw new LicenseInstallationFailedException(REPO_IS_LOCKED, e);
        } catch (LicenseViolationException e) {
            logger.info(INVALID);
            throw new LicenseInstallationFailedException(INVALID, e);
        } catch (IOException e) {
            logger.info(FS_ERROR);
            throw new LicenseInstallationFailedException(FS_ERROR, e);
        }
    }

    @Override
    public String getLicenseRenewalMessage() {
        if (!licenseFile.exists()) {
            return null;
        }

        try {
            licenseService.validate();
        } catch (LicenseVersionException e) {
            logger.info(LICENSE_STATUS, e.getMessage());
            logger.info(LICENSE_OLD_VERSION);
            return LICENSE_OLD_VERSION;
        } catch (LicensePeriodExpiredException e) {
            logger.info(LICENSE_STATUS, e.getMessage());
            logger.info(LICENSE_EXPIRED);
            return LICENSE_EXPIRED;
        } catch (LicenseEditionException e) {
            logger.info(LICENSE_STATUS, e.getMessage());
            License license = licenseService.getLicense();
            if (license != null && COMMUNITY.equals(license.getStringValue(LicenseProperty.EDITION))) {
                String message = format(LICENSE_COMMUNITY_MIX, licenseService.getProduct());
                logger.info(message);
                return message;
            }
            String editionMessage = LICENSE_ESSENTIALS_PRO_PREMIUM_OR_TRIAL;
            if (license != null && VERSION_3.equals(LicenseType.fromType(license.getStringValue(LicenseProperty.LICENSE_VERSION)))) {
                editionMessage = LICENSE_ENTERPRISE_OR_TRIAL;
            }
            String message = format(editionMessage, licenseService.getProduct());
            logger.info(message);
            return message;
        } catch (AmountOfCisExceededException e) {
            logger.info(LICENSE_STATUS, e.getMessage());
            logger.info(LICENSE_CI_EXCEEDS_LIMIT);
            return LICENSE_CI_EXCEEDS_LIMIT;
        } catch (UnregisteredLicenseExpiredException e) {
            logger.info(LICENSE_STATUS, e.getMessage());
            String message = format(LICENSE_UNREGISTERED_EXPIRED, licenseService.getProduct());
            logger.info(message);
            return message;
        } catch (LicenseViolationException e) {
            logger.info(LICENSE_STATUS, e.getMessage());
            String message = format(LICENSE_NOT_VALID, e.getMessage());
            logger.info(message);
            return message;
        }

        License license = licenseService.getLicense();
        if (license != null &&
                !license.isDummyLicense() &&
                license.getStringValue(LicenseProperty.EDITION).equals(LicenseVersion4.Edition4.Unregistered.name())) {
            logger.info(LICENSE_CONTINUE_TRIAL);
            return LICENSE_CONTINUE_TRIAL;
        } else {
            logger.info(LICENSE_ENTER_NEW);
            return LICENSE_ENTER_NEW;
        }
    }

    private License readLicense(String licenseText) throws IOException {
        File tempFile = File.createTempFile("license", "tmp");
        Files.write(tempFile.toPath(), licenseText.getBytes(UTF_8));
        LicenseReader licenseReader = new LicenseReader();
        return licenseReader.readLicense(tempFile);
    }
}
