package com.xebialabs.license.service.aws

import com.xebialabs.license.{License, LicenseReader, LicenseVerificationException}
import com.xebialabs.license.service.LicenseConfig
import com.xebialabs.license.service.aws.AWSLicenseEndpoint.AWSLicense
import grizzled.slf4j.Logging
import org.apache.http.client.config.RequestConfig
import org.apache.http.client.methods.{HttpGet, HttpPost}
import org.apache.http.client.utils.URIBuilder
import spray.json.DefaultJsonProtocol
import spray.json._

trait AWSLicenseEndpoint extends Logging with HttpClientAware {
  val licenseConfig: LicenseConfig
  import LicenseEndpointFormats._

  def validateProductCode(productCode: String): String = {
    logger.info(s"Validating AWS Marketplace product code (product code: $productCode)")
    val endpoint = licenseConfig.license.endpoint
    getHttpClient().execute(new HttpGet(s"$endpoint/api/marketplace/aws/$productCode"), responseHandler()) match {
      case Some(_) => productCode
      case None => throw new LicenseVerificationException(s"Product code '$productCode' is not valid.")
    }
  }

  def retrieveLicense(productCode: String, nonce: String, product: String): AWSLicense = {
    logger.info(s"Retrieving AWS Marketplace license (product code: $productCode, product: $product, nonce: $nonce)")
    val endpoint = licenseConfig.license.endpoint
    val uri = new URIBuilder(s"$endpoint/api/marketplace/aws/$productCode/$product").addParameter("nonce", nonce).build()
    getHttpClient().execute(new HttpPost(uri), responseHandler()) match {
      case Some(x) => x.parseJson.convertTo[AWSLicense]
      case None => throw new LicenseVerificationException(s"Could not retrieve AWS license...")
    }
  }

  override private[aws] def requestConfig = RequestConfig.DEFAULT
}

object LicenseEndpointFormats extends DefaultJsonProtocol {
  implicit val awsLicenseFormat: RootJsonFormat[AWSLicense] = jsonFormat1(AWSLicense)
}

object AWSLicenseEndpoint {
  case class AWSLicense(license: String) {
    def asLicense(): License = {
      new LicenseReader().readFromBase64(license)
    }
  }
}
