package com.xebialabs.license.service.aws

import java.util.UUID

import com.xebialabs.license.{License, LicenseProperty, LicenseVerificationException}
import com.xebialabs.license.service.{BaseLicenseService, LicenseCiCounterFactory}
import org.joda.time.Period

abstract class BaseAWSLicenseService extends BaseLicenseService {
  private[aws] var license: Option[License] = None
  val nonce: String = UUID.randomUUID().toString

  def getOrinitializeLicense: License

  override def getLicense: License = getOrinitializeLicense

  /**
    * Checks whether the license is expired
    */
  override def isLicenseExpired: Boolean = license.forall(_.isDateExpired)

  /**
    * Initialization that requires the repository
    */
  override def initialize(ciCounterFactory: LicenseCiCounterFactory): Unit = {
    this.ciCounterFactory = ciCounterFactory
    reLoadCiCounter(getOrinitializeLicense)

  }

  /**
    * Returns the amount of time this license is still valid
    *
    * @return a period. Returns period.ZERO if the license is expired
    */
  override def getValidPeriod: Period = Period.months(1)

  /**
    * Provide the product name.
    */
  override def getProduct: String = license.map(_.getStringValue(LicenseProperty.PRODUCT)).get

  /**
    * Reload the license file from the file system and {@link #validate()} it.
    */
  override def reload(): Unit = {
    val l = getOrinitializeLicense
    reLoadCiCounter(l)
    validate()
  }

  override def validate(license: License): Unit = {
    if (!license.hasLicenseProperty(LicenseProperty.AWS_NONCE)) {
      throw new LicenseVerificationException("The current license does not have the '" + LicenseProperty.AWS_NONCE + "' property set.")
    } else if (!license.getStringValue(LicenseProperty.AWS_NONCE).equals(nonce)) {
      throw LicenseAWSNonceException("The current license is for a different instance than your own.")
    }
    super.validate(license)
  }
}

case class AWSLicenseException(str: String) extends Exception(str)

