package com.xebialabs.license.service.aws

import com.xebialabs.license.{License, LicenseReader, LicenseVerificationException}
import com.xebialabs.license.service.LicenseConfig
import com.xebialabs.license.service.aws.AWSLicenseEndpoint.AWSLicense
import grizzled.slf4j.Logging
import org.apache.hc.core5.http.io.support.ClassicRequestBuilder
import org.apache.hc.core5.net.URIBuilder
import org.apache.hc.client5.http.config.RequestConfig
import org.apache.hc.core5.util.Timeout
import spray.json.DefaultJsonProtocol
import spray.json._

trait AWSLicenseEndpoint extends Logging with HttpClientAware {
  val licenseConfig: LicenseConfig
  import LicenseEndpointFormats._

  def validateProductCode(productCode: String): String = {
    logger.info(s"Validating AWS Marketplace product code (product code: $productCode)")
    val endpoint = licenseConfig.license.endpoint
    val request = ClassicRequestBuilder.get(s"$endpoint/api/marketplace/aws/$productCode").build()
    getHttpClient().execute(request, responseHandler()) match {
      case Some(_) => productCode
      case None => throw new LicenseVerificationException(s"Product code '$productCode' is not valid.")
    }
  }

  def retrieveLicense(productCode: String, nonce: String, product: String): AWSLicense = {
    logger.info(s"Retrieving AWS Marketplace license (product code: $productCode, product: $product, nonce: $nonce)")
    val endpoint = licenseConfig.license.endpoint
    val uri = new URIBuilder(s"$endpoint/api/marketplace/aws/$productCode/$product").addParameter("nonce", nonce).build()
    val request = ClassicRequestBuilder.post(uri).build()
    getHttpClient().execute(request, responseHandler()) match {
      case Some(x) => x.parseJson.convertTo[AWSLicense]
      case None => throw new LicenseVerificationException(s"Could not retrieve AWS license...")
    }
  }

  override private[aws] def requestConfig = {
    RequestConfig.custom()
    .setConnectionRequestTimeout(Timeout.ofMilliseconds(3000))
    .setResponseTimeout(Timeout.ofMilliseconds(5000))
    .build()
  }
}

object LicenseEndpointFormats extends DefaultJsonProtocol {
  implicit val awsLicenseFormat: RootJsonFormat[AWSLicense] = jsonFormat1(AWSLicense)
}

object AWSLicenseEndpoint {
  case class AWSLicense(license: String) {
    def asLicense(): License = {
      new LicenseReader().readFromBase64(license)
    }
  }
}