package com.xebialabs.plugin.manager.rest.json

import com.xebialabs.plugin.manager.metadata.{ArtifactId, MetadataEntry, PluginMetadata, Version}
import com.xebialabs.plugin.manager.rest.dto.PluginDto
import org.springframework.stereotype.Component
import spray.json._

import java.io.{IOException, OutputStream, OutputStreamWriter}
import java.lang.annotation.Annotation
import java.lang.reflect.Type
import javax.ws.rs.core.{MediaType, MultivaluedMap}
import javax.ws.rs.ext.{MessageBodyWriter, Provider}
import javax.ws.rs.{Produces, WebApplicationException}

@Component
@Provider
@Produces(Array(MediaType.APPLICATION_JSON))
class PluginDtoJsonWriter extends MessageBodyWriter[PluginDto]
  with BasicFormats
  with Version.Protocol
  with StandardFormats
  with MetadataEntry.Protocol {

  override def isWriteable(`type`: Class[_], genericType: Type, annotations: Array[Annotation],
                           mediaType: MediaType): Boolean = classOf[PluginDto].isAssignableFrom(`type`)

  override def getSize(viewList: PluginDto, `type`: Class[_], genericType: Type, annotations: Array[Annotation],
                       mediaType: MediaType): Long = -1

  @throws[IOException]
  @throws[WebApplicationException]
  override def writeTo(response: PluginDto, `type`: Class[_], genericType: Type, annotations: Array[Annotation],
                       mediaType: MediaType, httpHeaders: MultivaluedMap[String, AnyRef],
                       entityStream: OutputStream): Unit = {

    val writer = new OutputStreamWriter(entityStream)
    writer.write(response.pluginInfo.map(pluginsResult).toJson.compactPrint)
    writer.flush()
  }

  def pluginsResult(result: (ArtifactId, Option[PluginMetadata])): JsObject = result match {
    case (id, plugins) =>
      JsObject(
        "plugin" -> JsObject(
          "groupId" -> id.groupId.toJson,
          "artifactId" -> id.artifactId.toJson
        ),
        "versions" -> JsArray(JsObject(
          "version" -> id.version.toJson,
          "repository" -> id.repository.toJson
        )),
        "metadata" -> plugins.map(_.metadata).toJson,
        "hasLogo" -> plugins.exists(_.hasLogo).toJson
      )
  }

}
