package com.xebialabs.plugin.manager.cli

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.materializer.FilePlugin
import com.xebialabs.plugin.manager.sql.{DbPlugin, SqlPluginRepository}
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import org.apache.commons.io.FileUtils

import java.io.{BufferedReader, InputStreamReader}
import java.nio.file.{Files, Path, Paths}
import scala.jdk.CollectionConverters.IteratorHasAsScala
import scala.util.matching.Regex

class PluginManagerCli(pluginRepository: SqlPluginRepository, extension: String) {

  def run(): Unit = {
    ConfigWrapper.extension = extension

    val pluginsFromDb: Seq[DbPlugin] = pluginRepository.getAllWithoutBytes
    val pluginsFromFs: Seq[FilePlugin] = getFromFilesystem
    val context = new CommandContext(pluginsFromDb, pluginsFromFs)

    printCliIntro()
    printPlugins(pluginsFromDb, pluginsFromFs)
    printAvailableCommands()

    if (pluginsFromDb.nonEmpty || pluginsFromFs.nonEmpty) {
      val stdin = new BufferedReader(new InputStreamReader(System.in))

      while (context.isAcceptingCommands) {
        print("> ")
        val input = stdin.readLine
        val command = createCommand(input)
        command.executeIn(context)
        refresh(context)
      }
    } else println("------No plugins in database or file system------")

    println("------Shutting down Plugin Manager CLI------")
    System.exit(0)
  }

  class CommandContext (var pluginsFromDb: Seq[DbPlugin], var pluginsFromFs: Seq[FilePlugin]) {
    def isAcceptingCommands: Boolean = pluginsFromDb.nonEmpty || pluginsFromFs.nonEmpty
  }

  def refresh(context: CommandContext):Unit = {
    context.pluginsFromDb = pluginRepository.getAllWithoutBytes
    context.pluginsFromFs = getFromFilesystem
  }

  def createCommand(input: String): Command = {

    input match  {
      case ListCommand.regex => ListCommand()
      case DeleteCommand.regex(pluginName) => DeleteCommand(pluginName)
      case QuitCommand.regex => QuitCommand()
      case _ => UnknownCommand(input)
    }
  }

  def printCliIntro(): Unit = {
    println("")
    println("------Plugin Manager CLI------")
    println("")
    println("Use the Plugin Manager CLI to manually remove plugins from the database \nand file system.")
    println("")
    println("This tool is meant to fix plugin configurations and does not check whether \nthe plugin is still in use by the system.")
    println("")
    println("In cluster setups, the tool only removes files from the local file system.\nFiles need to be manually removed from other nodes.")
    println("")
  }

  def printAvailableCommands(): Unit = {
    println("")
    println("Available commands:")
    println(String.format("%10s %40s", "list", "- Shows available plugins"))
    println(String.format("%19s %63s", "delete <NAME>", "- Deletes a plugin from the database and local filesystem"))
    println(String.format("%10s %46s", "quit", "- Exits the Plugin Manager CLI"))
    println("")
  }

  def getFromFilesystem: Seq[FilePlugin] = {
    val pluginsDir: Path = Paths.get("plugins")
    Files.walk(pluginsDir)
      .filter(_.isFile)
      .filter(file => file.getFileName.toString.endsWith(ConfigWrapper.extension))
      .map(filePath => FilePlugin(filePath))
      .iterator().asScala.toSeq
  }

  def askForPluginToDelete(stdin: BufferedReader)(callback: String => Unit)(implicit dbPlugins: Seq[DbPlugin], fsPlugins: Seq[FilePlugin]): Unit = {
    println("Which plugin do you want to delete? (Please type full name without version)")
    print("> ")
    val userInput = stdin.readLine

    if (userInput.equalsIgnoreCase("q") || userInput.equalsIgnoreCase("quit")) {
      println("------Shutting down Plugin Manager CLI------")
      System.exit(0)
    }
    else {
      if (dbPlugins.exists(dbPlugin => dbPlugin.name.equals(userInput)) || fsPlugins.exists(fsPlugin => fsPlugin.name.equals(userInput))) {
        callback.apply(userInput)
      } else {
        println("This is not a valid plugin name.")
      }
    }
  }

  private def printPlugins(dbPlugins: Seq[DbPlugin], fsPlugins: Seq[FilePlugin]): Unit = {
    println(String.format("%-40s%-20s%-10s%-10s", "NAME", "VERSION", "DATABASE", "FILESYSTEM"))
    println("-------------------------------------------------------------------------------------------")

    if (dbPlugins.nonEmpty) {
      dbPlugins.sortWith(_.name < _.name).foreach(plugin => {
        val existInFs = fsPlugins.exists(fsPlugin => fsPlugin.name.equals(plugin.name) && fsPlugin.version.equals(plugin.version.getOrElse("")))
        println(String.format("%-40s%-20s%-10s%-14s", plugin.name, plugin.version.getOrElse(""), "+", if (existInFs) "+" else "-"))
      })
    }
    if (fsPlugins.nonEmpty) {
      fsPlugins.sortWith(_.name < _.name).foreach(plugin => {
        if (!dbPlugins.exists(fsPlugin => fsPlugin.name.equals(plugin.name) && fsPlugin.version.getOrElse("").equals(plugin.version))) {
          println(String.format("%-40s%-20s%-10s%-14s", plugin.name, plugin.version, "-", "+"))
        }
      })
    }
    if (dbPlugins.isEmpty && fsPlugins.isEmpty) {
      println("No plugins in database or filesystem")
    }

    println("-------------------------------------------------------------------------------------------")
  }

  trait Command {
    def executeIn(commandContext: CommandContext): Unit
  }

  case class ListCommand() extends Command {
    def executeIn(commandContext: CommandContext): Unit = printPlugins(commandContext.pluginsFromDb, commandContext.pluginsFromFs)
  }

  object ListCommand {
    val regex = "list"
  }

  case class DeleteCommand(pluginToDelete: String) extends Command {
    def executeIn(context: CommandContext): Unit = {
      val pluginExistsInDb = context.pluginsFromDb.exists(dbPlugin => dbPlugin.name.equals(pluginToDelete))
      val pluginExistsInFs = context.pluginsFromFs.exists(fsPlugin => fsPlugin.name.equals(pluginToDelete))

      if (pluginExistsInDb || pluginExistsInFs) {
        delete(context, pluginExistsInDb, pluginExistsInFs)
      } else {
        println("This is not a valid plugin name.")
      }
    }

    def delete(context: CommandContext, shouldDeleteFromDb: Boolean, shouldDeleteFromFs: Boolean): Unit ={
      if (shouldDeleteFromFs) {
        deleteFromFs(pluginToDelete, context.pluginsFromFs)
      }

      if (shouldDeleteFromDb) {
        deleteFromDb(pluginToDelete, context.pluginsFromDb)
      }

      println("Please verify and delete plugin file in other cluster members' plugins directory if needed")
    }

    def deleteFromFs(pluginToDelete: String, plugins: Seq[FilePlugin]): Unit = {
      FileUtils.forceDelete(plugins.filter(plugin => plugin.name == pluginToDelete).head.filePath)
      println(s"$pluginToDelete deleted from filesystem")
    }

    def deleteFromDb(pluginToDelete: String, plugins: Seq[DbPlugin]): Unit = {
      pluginRepository.delete(plugins.filter(plugin => plugin.name == pluginToDelete).head)
      println(s"$pluginToDelete deleted from database")
    }
  }

  object DeleteCommand {
    val regex: Regex = "delete (.+)".r
  }

  case class QuitCommand() extends Command {
    def executeIn(commandContext: CommandContext): Unit = System.exit(0)
  }

  object QuitCommand {
    val regex = "quit"
  }

  case class UnknownCommand(command: String) extends Command {
    def executeIn(commandContext: CommandContext): Unit = {
      println(s"Command $command not supported.")
      println("")
      printAvailableCommands()
    }
  }
}
