package com.xebialabs.plugin.manager.materializer

import com.xebialabs.plugin.manager.sql.DbPlugin

import java.nio.file.{Files, Path}
import scala.util.matching.Regex

case class FilePlugin(name: String,
                      version: String,
                      extension: String,
                      filePath: Path) {}

object FilePlugin {
  lazy val pattern: Regex = ("""(.+)-(\d.*)\.([a-z]*)""").r
  lazy val patternNoVersion: Regex = ("""(.+)\.([a-z]*)""").r
  lazy val patternVersionWithNightly: Regex = ("""(.+)-(\d.*-\d.*)\.([a-z]*)""").r

  def apply(file: Path): FilePlugin = {
    file.getFileName.toString match {
      case patternVersionWithNightly(name, version, extension) =>
        new FilePlugin(name, version, extension, file)
      case pattern(name, version, extension) =>
        new FilePlugin(name, version, extension, file)
      case patternNoVersion(name, extension) =>
        new FilePlugin(name, null, extension, file)
      case _ =>
        new FilePlugin(null, null, null, null)
    }
  }

  implicit class FilePluginOps(val fsPlugin: FilePlugin) extends AnyVal {
    def isLowerVersionThan(dbPlugin: DbPlugin): Boolean = VersionComparisonUtils.isLowerVersion(fsPlugin.version, dbPlugin.version.getOrElse("0"))

    def isSameVersionLike(dbPlugin: DbPlugin): Boolean = VersionComparisonUtils.isSameVersion(fsPlugin.version, dbPlugin.version.getOrElse("0"))

    def higherVersionExistsIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.exists(dbPlugin => dbPlugin.name.equals(fsPlugin.name) && fsPlugin.isLowerVersionThan(dbPlugin))

    def doesntExistIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.forall(dbPlugin => !fsPlugin.name.equals(dbPlugin.name))

    def isHigherVersionThanAMatchIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.exists(dbPlugin => fsPlugin.name.equals(dbPlugin.name) && dbPlugin.isLowerVersionThan(fsPlugin))

    def hasDifferentContentAndVersionThanAMatchIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.exists(dbPlugin => fsPlugin.name.equals(dbPlugin.name) && (isContentDifferent(fsPlugin.filePath, dbPlugin.checksum) || !fsPlugin.isSameVersionLike(dbPlugin)))

    def hasDifferentContentThanAVersionMatchIn(dbPlugins: Seq[DbPlugin]): Boolean = dbPlugins.exists(dbPlugin => fsPlugin.name.equals(dbPlugin.name) && fsPlugin.isSameVersionLike(dbPlugin) && isContentDifferent(fsPlugin.filePath, dbPlugin.checksum))

    def isContentDifferent(filePluginPath: Path, dbPluginChecksum: String): Boolean = {
      val filePluginChecksum = DbPlugin.calculateChecksum(Files.readAllBytes(filePluginPath))
      !filePluginChecksum.equals(dbPluginChecksum)
    }
  }
}
