package com.xebialabs.plugin.manager.sql

import com.xebialabs.deployit.checksum.DefaultChecksumAlgorithmProviderFactory
import com.xebialabs.deployit.util.JavaCryptoUtils
import com.xebialabs.plugin.manager.{Plugin, PluginId}
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.materializer.{FilePlugin, VersionComparisonUtils}
import com.xebialabs.plugin.manager.rest.api.PluginSource.LOCAL
import com.xebialabs.plugin.manager.rest.api.PluginStatus.READY_FOR_INSTALL
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}

import java.nio.file.{Files, Path}
import java.security.MessageDigest

case class DbPlugin(id: Long,
                    name: String,
                    version: Option[String],
                    extension: String,
                    groupId: String,
                    installationStatus: PluginStatus.Value,
                    source: PluginSource.Value,
                    checksum: String,
                    pluginBytesId: Option[Long],
                    bytes: Option[Array[Byte]])

object DbPlugin {

  def apply(plugin: Plugin): DbPlugin = {
    plugin.id match {
      case artifact: Artifact => fromArtifact(artifact, plugin.content.readAllBytes())
      case localFile: LocalFile => fromLocalFile(localFile, plugin.content.readAllBytes())
      case _ => throw new IllegalArgumentException("Not a proper plugin for conversion to a dbPlugin")
    }
  }

  def fromLocalFile(localFile: LocalFile, bytes: Array[Byte]): DbPlugin = {
    new DbPlugin(0, localFile.basename, Some(localFile.version.map(v => v.id).orNull), ConfigWrapper.extension, LOCAL.toString, READY_FOR_INSTALL, LOCAL, calculateChecksum(bytes), None, Some(bytes))
  }

  def fromArtifact(artifact: Artifact, bytes: Array[Byte]): DbPlugin = {
    new DbPlugin(0, artifact.artifactId, Some(artifact.version.id), ConfigWrapper.extension, artifact.groupId, READY_FOR_INSTALL, PluginSource.withName(artifact.repository), calculateChecksum(bytes), None, Some(bytes))
  }

  def calculateChecksum(bytes: Array[Byte]): String = {
    val hash: MessageDigest = DefaultChecksumAlgorithmProviderFactory.defaultComparisonAlgorithm.getMessageDigest
    hash.update(bytes)
    JavaCryptoUtils.digest(hash)
  }

  implicit class DbPluginOps(val dbPlugin: DbPlugin) extends AnyVal {
    def isLowerVersionThan(filePlugin: FilePlugin): Boolean = VersionComparisonUtils.isLowerVersion(dbPlugin.version.getOrElse("0"), filePlugin.version)

    def isSameVersionLike(filePlugin: FilePlugin): Boolean = VersionComparisonUtils.isSameVersion(dbPlugin.version.getOrElse("0"), filePlugin.version)

    def higherVersionExistsIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.exists(fsPlugin => dbPlugin.name.equals(fsPlugin.name) && dbPlugin.isLowerVersionThan(fsPlugin))

    def doesntExistIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.forall(fsPlugin => !fsPlugin.name.equals(dbPlugin.name))

    def isHigherVersionThanAMatchIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.exists(fsPlugin => dbPlugin.name.equals(fsPlugin.name) && fsPlugin.isLowerVersionThan(dbPlugin))

    def shouldReplaceAMatchIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.exists(fsPlugin => fsPlugin.name.equals(dbPlugin.name) && isFileCleanedUp(fsPlugin.filePath))

    def isFileCleanedUp(filePluginPath: Path): Boolean = {
      !Files.exists(filePluginPath)
    }

    def hasDifferentContentThanAVersionMatchIn(fsPlugins: Seq[FilePlugin]): Boolean = fsPlugins.exists(fsPlugin => dbPlugin.name.equals(fsPlugin.name) && dbPlugin.isSameVersionLike(fsPlugin) && isContentDifferent(fsPlugin.filePath, dbPlugin.checksum))

    def isContentDifferent(filePluginPath: Path, dbPluginChecksum: String): Boolean = {
      val filePluginChecksum = DbPlugin.calculateChecksum(Files.readAllBytes(filePluginPath))
      !filePluginChecksum.equals(dbPluginChecksum)
    }
  }

}
