package com.xebialabs.plugin.manager.rest.api

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata, Version}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.rest.dto.{PluginDto, PluginDtoList, RepositoryDto}
import com.xebialabs.plugin.manager.service.PluginService
import com.xebialabs.plugin.manager.{Plugin, PluginId}
import org.apache.commons.io.IOUtils
import org.jboss.resteasy.plugins.providers.multipart.{InputPart, MultipartFormDataInput}
import org.joda.time.format.DateTimeFormat
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.io.{InputStream, OutputStream}
import java.util
import java.util.Collections
import javax.ws.rs._
import javax.ws.rs.core.{MediaType, Response, StreamingOutput}
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.language.postfixOps
import scala.reflect.io.File
import scala.util.{Failure, Success, Try}

@Controller
class PluginResourceImpl @Autowired()(val pluginService: PluginService) extends AbstractSecuredResource with PluginResource {

  val DATE_TIME_FORMAT = "MM-dd-YYYY hh:mm:ss z"
  implicit val ec: ExecutionContext = pluginService.pluginManager.ec

  override def list(): Response = {
    checkPermission(ADMIN)
    Response.ok(
      PluginDtoList(pluginService.extend(pluginService.listInstalled())),
      MediaType.APPLICATION_JSON
    ).build()
  }

  override def installOrUpdate(input: MultipartFormDataInput, @QueryParam("pluginId") pluginId: String): Response = {
    def installOfficial(pluginId: PluginId, inputParts: util.List[InputPart]): Response = {
      inputParts.forEach { part =>
        val is = part.getBody[InputStream](classOf[InputStream], null)
        pluginService.install(Plugin(pluginId, None, is))
      }
      Response.ok().build()
    }

    def tryInstallLocal(pluginId: String, inputParts: util.List[InputPart]): Response = {
      Try(LocalFile(pluginId)) match {
        case Success(localFile) => installOfficial(localFile, inputParts)
          Response.ok().build()
        case Failure(exception) => Response
          .status(Response.Status.BAD_REQUEST)
          .entity(ValidationResult(exception.getMessage))
          .build()
      }
    }

    def tryInstall(pluginId: String, basename: String, version: String, inputParts: util.List[InputPart]): Response = {
      // first check if the plugin is found in metadata
      val searchResults = pluginService.searchOfficial(basename)
      // if it is, then it's an official plugin
      if (searchResults.size == 1) {
        val artifact = searchResults.head._1.toArtifact
        installOfficial(artifact.copy(version = Version.fromString(version).orNull), inputParts)
      } else {
        // if not, check if there is an official plugin with same name already installed in the system
        val installedOfficial = pluginService.listInstalled().collect { case a: Artifact => a }.filter(officialPlugin => officialPlugin.artifactId.equalsIgnoreCase(basename))
        // if there is, then it's an official plugin
        if (installedOfficial.size == 1) {
          val installed = installedOfficial.head
          val idString = installed.repository + ":" + installed.groupId + ":" + pluginId
          installOfficial(PluginId.Artifact.fromIdString(idString).get, inputParts)
        } else {
          // if not, then it's a 3rd party plugin
          tryInstallLocal(pluginId, inputParts)
        }
      }
    }

    checkPermission(ADMIN)
    val uploadForm = input.getFormDataMap
    val inputParts = uploadForm.get("file")

    pluginId match {
      case PluginId.patternVersionWithNightly(name, versionWithNightly) =>
        tryInstall(pluginId, name, versionWithNightly, inputParts)
      case PluginId.pattern(name, version) =>
        tryInstall(pluginId, name, version, inputParts)
      case _ =>
        tryInstallLocal(pluginId, inputParts)
    }
  }

  override def uninstall(repositoryId: String,
                         groupId: String,
                         artifactId: String,
                         version: String): Response = {
    checkPermission(ADMIN)

    val pluginSource = PluginSource.withName(repositoryId)
    val optionalVersion = if (version.isEmpty) None else Some(version.split("/")(1))

    pluginService.uninstall(pluginSource, groupId, artifactId, optionalVersion).map { _ =>
      Response.ok.build()
    }.fold (
      err => throw err,
      ok => ok
    )
  }

  override def search(query: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      PluginDtoList(pluginService.search(query).map(toPluginDto).toSeq),
      MediaType.APPLICATION_JSON
    ).build()
  }

  override def listRepositories(): Response = {
    checkPermission(ADMIN)
    Response.ok(RepositoryDto(pluginService.repositories.values.toSeq), MediaType.APPLICATION_JSON).build()
  }

  override def updateRepository( repositoryId: String): Response = {
    checkPermission(ADMIN)
    val lastDownloaded = withRepo(repositoryId)(_.update().map(DateTimeFormat.forPattern(DATE_TIME_FORMAT).print(_)))
    Response.ok(
      Collections.singletonMap("lastDownloaded", lastDownloaded)
    ).build()
  }

  override def listRepository(repositoryId: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      PluginDtoList(withRepo(repositoryId)(_.list()).map(toPluginDto).toSeq),
      MediaType.APPLICATION_JSON
    ).build()
  }

  override def installOrUpdateFromRepository(repositoryId: String,
                                             groupId: String,
                                             artifactId: String,
                                             version: String): Response = {
    checkPermission(ADMIN)
    val artifact = PluginId.Artifact.apply(repositoryId, groupId, artifactId, version)
    pluginService.installFromRepository(artifact).map { _ =>
      Response.ok(artifact, MediaType.APPLICATION_JSON).build()
    }.fold(
      err => throw err,
      ok => ok
    )
  }

  override def getLogo(repositoryId: String, groupId: String, artifactId: String): Response = {
    checkPermission(ADMIN)
    pluginService.getLogo(repositoryId, artifactId).map { logoFile =>
      val input = File(logoFile).inputStream()
      val stream = new StreamingOutput {
        override def write(output: OutputStream): Unit = {
          output.write(IOUtils.toByteArray(input))
          input.close()
        }
      }
      Response.ok(stream, MediaType.APPLICATION_OCTET_STREAM).build()
    }.getOrElse {
      throw new NotFoundException(s"Couldn't find logo for '$groupId:$artifactId' in repository '$repositoryId'")
    }
  }

  private def withRepo[T](repositoryId: String, duration: Duration = 30 seconds)(action: PluginsRepository => Future[T]): T = {
    withRepo0(repositoryId) { repository =>
      Await.result(action(repository), duration)
    }
  }

  private def withRepo0[T](repositoryId: String)(action: PluginsRepository => T): T = {
    pluginService.repositories.get(repositoryId) match {
      case None =>
        throw new NotFoundException(s"Unknown plugin repository '$repositoryId'")

      case Some(repository) =>
        action(repository)
    }
  }

  def toPluginDto: ((ArtifactId, Option[PluginMetadata])) => PluginDto = tuple => PluginDto(tuple._1, tuple._2)

}
