package com.xebialabs.plugin.manager

import java.nio.file.{FileAlreadyExistsException, Files, Paths}
import java.util.stream.Collectors

import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.plugin.manager.repository.PluginRepository

import scala.collection.JavaConverters._
import scala.reflect.io.File
import scala.util.matching.Regex

trait PluginManager extends PluginRepository {

  def install(plugin: Plugin): Unit

  def uninstall(metadata: PluginMetadata): Unit

}

class LocalPluginManager() extends PluginManager {

  private val folder: String = "plugins"

  private val plugins = File(folder)

  override val name: String = "local"

  plugins.createDirectory(force = true, failIfExists = false)

  val pattern = new Regex("(.*)-(\\d.*)\\.jar")

  override def list(): Seq[PluginMetadata] = {
    Files.list(Paths.get(folder))
      .map[String](_.getFileName.toString)
      .filter(_.endsWith(".jar"))
      .map[PluginMetadata](PluginMetadata(_))
      .collect(Collectors.toList[PluginMetadata]).asScala
      .sortBy(_.fileName)
  }

  override def get(pluginMetadata: PluginMetadata): Plugin = ???

  override def install(plugin: Plugin): Unit = {
    try {
      uninstallPreviousPlugin(plugin)
      Files.copy(
        plugin.content,
        Paths.get(s"$folder${System.getProperty("file.separator")}${plugin.metadata.fileName}"))
    } catch {
      case e: FileAlreadyExistsException =>
        throw new ItemConflictException("A plugin named '%s' is already installed.", plugin.metadata.fileName)
    } finally {
      plugin.content.close()
    }
  }

  override def uninstall(metadata: PluginMetadata): Unit =
    Files.deleteIfExists(Paths.get(s"$folder${System.getProperty("file.separator")}${metadata.fileName}"))

  private def uninstallPreviousPlugin(plugin: Plugin): Unit = {
    val foundIterator = pattern.findAllIn(plugin.metadata.fileName)
    if (foundIterator.nonEmpty) {
      val fileName = foundIterator.group(1)
      val found = list().find(p => pattern.findAllIn(p.fileName).group(1) == fileName)
      found.foreach(uninstall)
    }
  }
}
