package com.xebialabs.plugin.manager.service

import java.io.File

import com.typesafe.config.{Config, ConfigObject}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig, NexusServerConfig}
import com.xebialabs.plugin.manager.service.PluginService.defaultTimeout
import com.xebialabs.plugin.manager.{Plugin, PluginId, PluginManager}
import grizzled.slf4j.Logging
import javax.annotation.PreDestroy

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.language.postfixOps
import scala.util.{Failure, Try}


trait PluginService extends Logging {

  def pluginManager: PluginManager

  implicit lazy val ec: ExecutionContext = pluginManager.ec

  def repositories: mutable.Map[String, PluginsRepository]

  def addRepository(repository: PluginsRepository): Boolean

  def deleteRepository(name: String): Boolean


  def update(): Unit = {
    if (repositories.nonEmpty) {
      logger.info(s"Updating all (${repositories.size}) repositories...")
      Await.ready(Future.sequence(repositories.values.map(_.update())), defaultTimeout)
    }
  }

  def search(query: Option[String]): Map[ArtifactId, Option[PluginMetadata]] =  {
    logger.debug(s"search($query)")
    Await.result(
      for {
        installed <- pluginManager.search(query)
        available <- Future.sequence(repositories.values.map(_.search(query))).map(_.flatten)
      } yield {
        available.foldLeft(installed.map(_ -> Option.empty[PluginMetadata]).toMap) {
          case (acc, (id, pm1)) =>
            acc.get(id) match {
              case None =>
                acc.updated(id, pm1)
              case Some(_) =>
                pm1.map(pm => acc.updated(id, Some(pm))).getOrElse(acc)
            }
        }
      },
      defaultTimeout
    )
  }

  def search(query: String): Map[ArtifactId, Option[PluginMetadata]] = search(Option(query).filter(_.nonEmpty))

  def list(): Map[ArtifactId, Option[PluginMetadata]] = search(None)

  def listInstalled(): Seq[PluginId] =
    pluginManager.listInstalled()

  def install(plugin: Plugin): Unit = {
    logger.info(s"Installing plugin ${plugin.id.id}...")
    pluginManager.install(plugin)
  }

  def installFromRepository(id: PluginId.Artifact): Try[Unit] =
    repositories.get(id.repository).map { repo =>
      Try {
        val plugin = Await.result(repo.get(id), defaultTimeout)
        install(plugin)
      }.recoverWith {
        case err =>
          logger.warn(err.getMessage)
          Failure(err)
      }
    }.getOrElse {
      Failure(new NotFoundException(s"Unknown plugin repository '${id.repository}"))
    }

  def uninstall(id: PluginId): Boolean = {
    logger.info(s"Uninstalling plugin $id")
    pluginManager.uninstall(id)
  }

  def getLogo(repositoryId: String, artifactId: String): Option[File] =
    for {
      repo <- repositories.get(repositoryId)
      logo <- repo.getLogo(artifactId)
    } yield logo

  def attachMetadata: ArtifactId => (ArtifactId, Option[PluginMetadata]) = artifactId => {
      artifactId -> {
        for {
          repoId <- artifactId.repository
          repo <- repositories.get(repoId)
          pluginsMeta <- repo.getMetadata(artifactId)
        } yield pluginsMeta
      }
  }

  def extend(data: Seq[PluginId]): Map[ArtifactId, Option[PluginMetadata]] =
    data.map(_.toArtifactId).toSet.map(attachMetadata).toMap


  @PreDestroy
  def shutdown(): Unit = {
    logger.info("Shutting down LocalPluginManager...")
    Await.ready(
      Future.sequence(repositories.values.collect {
        case nexus: NexusPluginRepository => nexus.shutdown()
      }),
      repositories.size * defaultTimeout
    )
  }

}


object PluginService {
  // TODO: get this from configuration
  val defaultTimeout: Duration = 180 seconds

  def serversFromConfig(pluginsConfig: Config): Map[String, Try[NexusServerConfig]] =
    Try(pluginsConfig.getObject("servers")).map(
      _.asScala.collect {
        case (serverName, serverConfig) =>
          serverName -> NexusServerConfig.fromConfig(serverConfig.asInstanceOf[ConfigObject].toConfig)
      }.toMap
    ).getOrElse(Map.empty)

  def repositoriesFromConfig(pluginsConfig: Config, servers: Map[String, Try[NexusServerConfig]])
                            (implicit productConfig: ProductConfig): Map[String, Try[NexusRepositoryConfig]] =
    Try(pluginsConfig.getObject("repositories")).map(
      _.asScala
        .map { case (name, repoConfig) =>
          name -> NexusRepositoryConfig.fromConfig(name)(repoConfig.asInstanceOf[ConfigObject].toConfig, servers)
        }.toMap
    ).getOrElse(Map.empty)

  def configuredRepositories(pluginsConfig: Config)
                            (implicit productConfig: ProductConfig): List[Try[NexusPluginRepository]] =
    repositoriesFromConfig(pluginsConfig, serversFromConfig(pluginsConfig))
      .map { case (name, tryConfig) =>
        tryConfig.map(config => NexusPluginRepository.memCached(name, config))
      }.toList
}