package com.xebialabs.plugin.manager.cli

import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import org.kohsuke.args4j.spi.StringArrayOptionHandler
import org.kohsuke.args4j.{CmdLineException, CmdLineParser, Option, ParserProperties}

import java.io.PrintStream
import java.nio.file.{Path, Paths}
import java.util
import scala.annotation.tailrec
import scala.jdk.javaapi.CollectionConverters
import scala.language.implicitConversions

/**
 *
 */
class PluginManagerCliLaunchOptions {
  @Option(name = "-list", usage = "Shows available plugins")
  var list: Boolean = false

  @Option(name = "-add", usage = "Adds a plugin to the database and local filesystem", metaVar = "PLUGIN_FILE")
  var add: String = _

  @Option(name = "-update", usage = "Updates a plugin to the database and local system", metaVar = "PLUGIN_NAME PLUGIN_FILE",
    handler = classOf[StringArrayOptionHandler])
  var update: util.List[String] = _

  @Option(name = "-delete", usage = "Deletes a plugin from the database and local filesystem. " +
    "If no version is specified, delete will be successful only if one plugin exists",
    metaVar = "PLUGIN_NAME [PLUGIN_VERSION]", handler = classOf[StringArrayOptionHandler])
  var delete: util.List[String] = _

  @Option(name = "-help", usage = "Prints usage message", help = true)
  var printUsage: Boolean = false
}

object PluginManagerCliLaunchOptions {

  def parseCommandLine(args: Array[String]): PluginManagerCliLaunchOptions = {
    def emptyOptions(options: PluginManagerCliLaunchOptions) = {
      options.add == null &&
        !options.list &&
        options.update == null &&
        options.delete == null &&
        !options.printUsage
    }

    val options = new PluginManagerCliLaunchOptions

    val properties = ParserProperties.defaults.withShowDefaults(false)
    val parser = new CmdLineParser(options, properties)

    try {
      parser.parseArgument(CollectionConverters.asJavaCollection(args))
      if (options.printUsage || emptyOptions(options)) {
        printUsage(parser, System.out)
        return null
      }
    } catch {
      case e: CmdLineException =>
        System.err.println(e.getMessage)
        printUsage(parser, System.err)
        return null
    }
    options
  }

  private def printUsage(parser: CmdLineParser, stream: PrintStream): Unit = {
    stream.println("Supported options:")
    parser.printUsage(stream)
  }

  implicit class UtilListOps(list: util.List[String]) {
    def validate: UpdateOp =
      if (list.size() == 2) {
        val pluginName = list.get(0)
        val pluginToAdd = list.get(1)
        val pluginPath = getPluginPathNormalized(Paths.get(pluginToAdd))
        pluginPath match {
          case None => PluginAtPathDoesntExist(pluginToAdd)
          case Some(path) => Valid(pluginName, path)
        }
      }
      else WrongUsage()
  }

  @tailrec
  def getPluginPathNormalized(pluginToAdd: Path): scala.Option[Path] = {
    val currentDirectory = Paths.get(System.getProperty("user.dir"))
    if ((pluginToAdd.isAbsolute && pluginToAdd.exists()) || currentDirectory.resolve(pluginToAdd).exists()) {
      scala.Option(pluginToAdd)
    } else if (pluginToAdd.getNameCount == 1) {
      None
    } else {
      getPluginPathNormalized(pluginToAdd.subpath(1, pluginToAdd.getNameCount))
    }
  }
}

class UpdateOp(pluginName: String, pluginToAdd: String, pluginPath: scala.Option[Path])

case class WrongUsage() extends UpdateOp(pluginName = "", pluginToAdd = "", pluginPath = None)

case class PluginAtPathDoesntExist(pluginToAdd: String) extends UpdateOp(pluginName = "", pluginToAdd = pluginToAdd, pluginPath = None)

case class Valid(pluginName: String,  path: Path) extends UpdateOp(pluginName = pluginName, pluginToAdd = "", pluginPath = Some(path))