package com.xebialabs.plugin.manager.repository.nexus

import akka.actor.ActorSystem
import akka.http.scaladsl.model.MediaRanges.`text/*`
import akka.http.scaladsl.model.MediaTypes.{`application/json`, `application/octet-stream`}
import akka.http.scaladsl.model.Multipart.FormData
import akka.http.scaladsl.model._
import akka.http.scaladsl.model.headers.{Accept, Authorization, Expect}
import akka.stream.Materializer
import akka.stream.scaladsl.StreamConverters
import akka.util.ByteString
import com.xebialabs.plugin.manager.repository.nexus.NexusPluginRepositoryFixturesException.{DeleteError, UploadError}
import grizzled.slf4j.Logging

import scala.concurrent.{ExecutionContext, Future}
import scala.language.postfixOps


class NexusRepositoryManager(name: String, server: NexusServerConfig)(implicit m: Materializer, ec: ExecutionContext, as: ActorSystem)
  extends NexusJsonProtocol with Logging {

  def addArtifacts(repository: String, artifacts: NexusArtifact*): Future[Seq[ArtifactMetadata]] =
    Future.sequence(artifacts.map(a => addArtifact(repository, a)))

  def addJsonArtifacts(repository: String, artifacts: NexusArtifact*): Future[Seq[ArtifactMetadata]] =
    Future.sequence(artifacts.map(a => addJsonArtifact(repository, a)))

  def addJsonArtifact(repository: String, artifact: NexusArtifact): Future[ArtifactMetadata] =
    addArtifact(repository, artifact, Some(`application/json`))

  def addArtifact(repository: String, artifact: NexusArtifact, explicitContentType: Option[ContentType] = None): Future[ArtifactMetadata] = {
    val entity = uploadEntity(artifact, repository, explicitContentType)

    logger.info(s"$name: Uploading artifact ${artifact.metadata.id}")

    for {
      response <- uploadArtifact(entity)
      artifactMetadata <- response
        .onSuccess(_.asJson[ArtifactMetadata])
        .onFailure((status, msg) => UploadError(artifact.metadata, status, msg))
    } yield {
      logger.info(s"$name: Uploaded artifact to ${server.uri}: ${artifact.metadata.id}")
      artifactMetadata
    }
  }

  def removeArtifact(repository: String, metadata: ArtifactMetadata): Future[Unit] = {
    logger.trace(s"$name: removeArtifact($repository, $metadata)}")
    for {
      resp <- deleteArtifact(repository, metadata)
      _ <- resp
        .onSuccess(_.asString)
        .onFailure((status, msg) => DeleteError(metadata, status, msg))
    } yield {
      logger.trace(s"$name: removed artifact ${metadata.id} from '$repository' repository.")
    }
  }

  protected def uploadEntity(artifact: NexusArtifact, repository: String, explicitContentType: Option[ContentType] = None): RequestEntity = {
    val metadata = artifact.metadata
    val parts = jsonParts(
      "r" -> repository,
      "a" -> metadata.artifactId,
      "g" -> metadata.groupId,
      "v" -> metadata.version.id,
      "hasPom" -> "false",
      "e" -> metadata.packaging,
      "p" -> metadata.packaging
    )
    val classifier = artifact.classifier.fold(Seq.empty[Multipart.FormData.BodyPart])(c => jsonParts("c" -> c))

    val contentType: ContentType = explicitContentType.getOrElse(`application/octet-stream`)

    val content: FormData.BodyPart = artifact.content match {
      case Left(path) =>
        Multipart.FormData.BodyPart.fromPath("file", contentType, path)

      case Right(Left(input)) =>
        Multipart.FormData.BodyPart("file",
          HttpEntity.IndefiniteLength(contentType, StreamConverters.fromInputStream(() => input)),
          Map("filename" -> metadata.fileName)
        )

      case Right(Right(data)) =>
        Multipart.FormData.BodyPart("file",
          HttpEntity(contentType, ByteString(data.getBytes)),
          Map("filename" -> metadata.fileName)
        )

    }
    Multipart.FormData(parts ++ classifier :+ content: _*).toEntity()
  }

  protected def uploadArtifact(entity: RequestEntity): Future[HttpResponse] = {
    val uri = server.uri.withPath(server.serviceUri.path / "artifact" / "maven" / "content")
    server.httpRequest(
      HttpRequest(HttpMethods.POST, uri = uri, entity = entity, headers = List(
        Authorization(server.credentials),
        Accept(`text/*`),
        Expect.`100-continue`
      ))
    )
  }

  protected def jsonParts(kvs: (String, String)*): Seq[Multipart.FormData.BodyPart] = kvs.map { case (k, v) =>
    Multipart.FormData.BodyPart(k, v)
  }

  protected def deleteArtifact(repository: String, metadata: ArtifactMetadata): Future[HttpResponse] = {
    val groupPath = metadata.groupId.split('.').foldLeft[Uri.Path](Uri.Path.Empty)(_ / _)
    val artifactUri = server.uri.withPath(server.serviceUri.path / "repositories" / repository / "content" ++ groupPath / metadata.artifactId)
    for {
      deleteResponse <- server.httpRequest(
        HttpRequest(HttpMethods.DELETE, uri = artifactUri, headers = List(
          Authorization(server.credentials),
          Accept(`text/*`)
        ))
      )
    } yield deleteResponse
  }
}


sealed abstract class NexusPluginRepositoryFixturesException(message: String) extends Exception(message)

object NexusPluginRepositoryFixturesException {

  case class UploadError(metadata: ArtifactMetadata, status: StatusCode, message: String) extends NexusPluginRepositoryFixturesException(
    s"Error uploading artifact ${metadata.id}: $status: $message"
  )

  case class DeleteError(metadata: ArtifactMetadata, status: StatusCode, message: String) extends NexusPluginRepositoryFixturesException(
    s"Error deleting artifact ${metadata.id}: $status: $message"
  )

}