package com.xebialabs.plugin.manager.rest.api;

import com.xebialabs.xlplatform.documentation.PublicApi;
import org.jboss.resteasy.plugins.providers.multipart.MultipartFormDataInput;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;

/**
 * API for managing plugins. User must have ADMIN permission level in order to interact with the plugin manager API.
 * The API exposes two main flows: installing or upgrading an officially supported Digital.ai plugin which is hosted
 * over the official Nexus repository; and installing or upgrading an official or 3rd party plugin xldp via manual upload.
 */
@Path("/plugin-manager")
@Produces({APPLICATION_JSON})
@PublicApi
public interface PluginResource {


    /**
     * Lists all plugins in the system (either in INSTALLED or READY_FOR_INSTALL status) with metadata for each plugin.
     *
     * @return a list of plugins in the system
     * @restDetails Returns a response with a status code of 200 (OK) and a list of plugins with all details.
     * @apiDetails Returns a response with a status code of 200 (OK) and a list of plugins with all details.
     * @permission Available only to admin
     */
    @GET
    @Path("list")
    @Produces({MediaType.APPLICATION_JSON})
    Response list();

    /**
     * Lists all configured Nexus repositories in order to later query for available official plugins.
     * Note that the response field `config.repository-id` is a mandatory parameter in some other API calls.
     * The default repository id is xld-official.
     *
     * @return a list of nexus repository objects from server configuration
     * @restDetails Returns a response with a status code of 200 (OK) and a list of repositories with all details.
     * @apiDetails Returns a response with a status code of 200 (OK) and a list of repositories with all details.
     * @permission Available only to admin
     */
    @GET
    @Path("repositories")
    @Produces({MediaType.APPLICATION_JSON})
    Response listRepositories();

    /**
     * Lists all available plugins for the provided repositoryId. Returns metadata and installation status per plugin.
     * Server internally caches the received metadata so an update API call (`POST /plugin-manager/repositories/{repository-id}/update`) needs to be made periodically:.
     *
     * @param repositoryId the Nexus repository ID on which plugin is accessible (in case of 3rd party plugins it should be $$__local__$$); for example, xld-official
     * @return a list of officially supported plugin objects for the given repository
     * @restDetails Returns a response with a status code of 200 (OK) and a list of all available official plugins for
     * the provided repositoryId. If provided repositoryId is wrong then returns 404 (NOT FOUND).
     * @apiDetails Returns a response with a status code of 200 (OK) and a list of all available official plugins for
     * the provided repositoryId. If provided repositoryId is wrong then returns 404 (NOT FOUND).
     * @permission Available only to admin
     */
    @GET
    @Path("repositories/{repositoryId}/list")
    @Produces({MediaType.APPLICATION_JSON})
    Response listRepository(@PathParam("repositoryId") String repositoryId);

    /**
     * Downloads a plugin from repository and inserts it into the DB. Plugin is then in the READY_FOR_INSTALL status,
     * and a system restart is needed for installation to be completed. The endpoint is for updating a plugin as well.
     * The update scenario will get triggered if the plugin with passed repositoryId, groupId and artifactId is already
     * in the system.
     *
     * Note: the endpoint can't be used for downgrading, only for upgrades.
     *
     * @param repositoryId the Nexus repository ID on which plugin is accessible (in case of 3rd party plugins it should be $$__local__$$); for example, xld-official
     * @param groupId      the coordinates of plugin on the Nexus repository (in case of 3rd party plugins it should be $$__local__$$); for example, com.xebialabs.deployit.plugins
     * @param artifactId   the name of artifact on the Nexus repository; for example, xld-docker-plugin
     * @param version      the version of the artifact which should be installed; for example, 10.1.0
     * @return information on the installed plugin
     * @restDetails Returns a response with a status code of 200 (OK) and installed plugin information. If provided
     * repositoryId is wrong then returns 404 (NOT FOUND). If there is an issue with plugin installation
     * then returns 500 (INTERNAL SERVER ERROR).
     * @apiDetails Returns a response with a status code of 200 (OK) and installed plugin information. If provided
     * repositoryId is wrong then returns 404 (NOT FOUND). If there is an issue with plugin installation
     * then returns 500 (INTERNAL SERVER ERROR).
     * @permission Available only to admin
     */
    @POST
    @Path("repositories/{repositoryId}/install-or-update/{groupId}/{artifactId}/{version}")
    @Produces({MediaType.APPLICATION_JSON})
    @Deprecated
    Response installOrUpdateFromRepository(@PathParam("repositoryId") String repositoryId,
                                           @PathParam("groupId") String groupId,
                                           @PathParam("artifactId") String artifactId,
                                           @PathParam("version") String version);

    /**
     * Downloads a plugin from repository and inserts it into the DB. Plugin is then in the READY_FOR_INSTALL status,
     * and a system restart is needed for installation to be completed.
     *
     * @param repositoryId the Nexus repository ID on which plugin is accessible (in case of 3rd party plugins it should be $$__local__$$); for example, xld-official
     * @param groupId      the coordinates of plugin on the Nexus repository (in case of 3rd party plugins it should be $$__local__$$); for example, com.xebialabs.deployit.plugins
     * @param artifactId   the name of artifact on the Nexus repository; for example, xld-docker-plugin
     * @param version      the version of the artifact which should be installed; for example, 10.1.0
     * @return information on the installed plugin
     * @restDetails Returns a response with a status code of 200 (OK) and installed plugin information. If provided
     * repositoryId is wrong then returns 404 (NOT FOUND). If there is an issue with plugin installation
     * then returns 500 (INTERNAL SERVER ERROR).
     * @apiDetails Returns a response with a status code of 200 (OK) and installed plugin information. If provided
     * repositoryId is wrong then returns 404 (NOT FOUND). If there is an issue with plugin installation
     * then returns 500 (INTERNAL SERVER ERROR).
     * @permission Available only to admin
     */
    @POST
    @Path("repositories/{repositoryId}/install/{groupId}/{artifactId}/{version}")
    @Produces({MediaType.APPLICATION_JSON})
    Response installFromRepository(@PathParam("repositoryId") String repositoryId,
                                   @PathParam("groupId") String groupId,
                                   @PathParam("artifactId") String artifactId,
                                   @PathParam("version") String version);

    /**
     * Updates a plugin from repository and updates the DB. Plugin is then in the INSTALLED status,
     * and a system restart is needed for installation to be completed.
     * The update scenario will get triggered if the plugin with passed repositoryId, groupId and artifactId is already
     * in the system.
     *
     * Note: the endpoint can't be used for downgrading, only for upgrades.
     *
     * @param repositoryId the Nexus repository ID on which plugin is accessible (in case of 3rd party plugins it should be $$__local__$$); for example, xld-official
     * @param groupId      the coordinates of plugin on the Nexus repository (in case of 3rd party plugins it should be $$__local__$$); for example, com.xebialabs.deployit.plugins
     * @param artifactId   the name of artifact on the Nexus repository; for example, xld-docker-plugin
     * @return information on the updated plugin
     * @restDetails Returns a response with a status code of 200 (OK) and updated plugin information. If provided
     * repositoryId is wrong then returns 404 (NOT FOUND). If there is an issue with plugin installation
     * then returns 500 (INTERNAL SERVER ERROR).
     * @apiDetails Returns a response with a status code of 200 (OK) and updated plugin information. If provided
     * repositoryId is wrong then returns 404 (NOT FOUND). If there is an issue with plugin update
     * then returns 500 (INTERNAL SERVER ERROR).
     * @permission Available only to admin
     */
    @POST
    @Path("repositories/{repositoryId}/update/{groupId}/{artifactId}")
    @Produces({MediaType.APPLICATION_JSON})
    Response updateFromRepository(@PathParam("repositoryId") String repositoryId,
                                  @PathParam("groupId") String groupId,
                                  @PathParam("artifactId") String artifactId);

    /**
     * Installs a plugin by uploading the plugin binary. It can either be an official plugin or a 3rd party plugin.
     * Plugin installation requires a restart of the system to be completed.
     *
     * Endpoint has different behavior depending on the type of the plugin uploaded, official or local.
     *
     * Local: if a plugin already exists in the system, the one in the system will get overwritten on next system start.
     * This will happen regardless of the plugin version (if it exists at all). I.e. local plugin upgrades and downgrades are allowed without any limitations.
     *
     * Official: if a plugin already exists in the system and the uploaded version is equal or higher than the plugin in the system then the system will overwrite the previous plugin on next system start.
     * If the uploaded version is lower than the one in the system the uploaded binary is ignored. I.e. it's possible to use this endpoint for official plugin upgrades, but it's impossible to use it for downgrades.
     *
     * The system will treat the uploaded file as an official plugin if the uploaded binary filename matches an entry in the official metadata.
     * Match is searched for using artifactId only; version and extension are ignored.
     *
     * If the plugin with the same name exists in the system, the system will replace the previous plugin, and it will
     * do so regardless of the version (if it exists at all).
     *
     * @param input the plugin file data
     * @param pluginId plugin filename; for example, xld-docker-plugin-9.2.0.xldp
     * @return empty in case of successful installation or error message in case of failure
     * @restDetails Returns an empty response with a status code of 200 (OK). In case of installation failure returns
     * a status code of 400 (BAD REQUEST) with a response object containing the error message.
     * @apiDetails Returns an empty response with a status code of 200 (OK). In case of installation failure returns
     * a status code of 400 (BAD REQUEST) with a response object containing the error message.
     * @permission Available only to admin
     */
    @POST
    @Path("install-or-update")
    @Consumes({"multipart/form-data"})
    @Deprecated
    Response installOrUpdate(MultipartFormDataInput input, @QueryParam("pluginId") String pluginId);

    /**
     * Installs a plugin by uploading the plugin binary. It can either be an official plugin or a 3rd party plugin.
     * Plugin installation requires a restart of the system to be completed.
     *
     * Endpoint has different behavior depending on the type of the plugin uploaded, official or local.
     *
     * The system will treat the uploaded file as an official plugin if the uploaded binary filename matches an entry in the official metadata.
     * Match is searched for using artifactId only; version and extension are ignored.
     *
     * @param input    the plugin file data
     * @param pluginId plugin filename; for example, xld-docker-plugin-9.2.0.xldp
     * @return empty in case of successful installation or error message in case of failure
     * @restDetails Returns an empty response with a status code of 200 (OK). In case of installation failure returns
     * a status code of 400 (BAD REQUEST) with a response object containing the error message.
     * @apiDetails Returns an empty response with a status code of 200 (OK). In case of installation failure returns
     * a status code of 400 (BAD REQUEST) with a response object containing the error message.
     * @permission Available only to admin
     */
    @POST
    @Path("install")
    @Consumes({"multipart/form-data"})
    Response install(MultipartFormDataInput input, @QueryParam("pluginId") String pluginId);

    /**
     * Updates a plugin by uploading the plugin binary. It can either be an official plugin or a 3rd party plugin.
     * Plugin update requires a restart of the system to be completed.
     *
     * Endpoint has different behavior depending on the type of the plugin uploaded, official or local.
     *
     * Local: if a plugin already exists in the system, the one in the system will get overwritten on next system start.
     * This will happen regardless of the plugin version (if it exists at all). I.e. local plugin upgrades and downgrades are allowed without any limitations.
     *
     * Official: if a plugin already exists in the system and the uploaded version is equal or higher than the plugin in the system then the system will overwrite the previous plugin on next system start.
     * If the uploaded version is lower than the one in the system the uploaded binary is ignored. I.e. it's possible to use this endpoint for official plugin upgrades, but it's impossible to use it for downgrades.
     *
     * The system will treat the uploaded file as an official plugin if the uploaded binary filename matches an entry in the official metadata.
     * Match is searched for using artifactId only; version and extension are ignored.
     *
     * @param input      the plugin file data
     * @param pluginId   plugin filename; for example, xld-docker-plugin-9.2.0.xldp
     * @param newVersion new version of the plugin
     * @return empty in case of successful update or error message in case of failure
     * @restDetails Returns an empty response with a status code of 200 (OK). In case of update failure returns
     * a status code of 400 (BAD REQUEST) with a response object containing the error message.
     * @apiDetails Returns an empty response with a status code of 200 (OK). In case of update failure returns
     * a status code of 400 (BAD REQUEST) with a response object containing the error message.
     * @permission Available only to admin
     */
    @POST
    @Path("update")
    @Consumes({"multipart/form-data"})
    Response update(MultipartFormDataInput input, @QueryParam("existingPluginName") String existingPluginName, @QueryParam("newPluginId") String newPluginId);

    /**
     * Cancels pending installation of a plugin. Plugin installation requires a restart of the system to be completed.
     * The uninstall API call can be made on plugins in the READY_FOR_INSTALL status (which is the state of plugin after
     * the install API call but before system restart).
     * Note: once a plugin is in INSTALLED state it can not be deleted anymore.
     *
     * @param repositoryId the Nexus repository ID on which plugin is accessible (in case of 3rd party plugins it should be $$__local__$$); for example, xld-official
     * @param groupId      the coordinates of plugin on the Nexus repository (in case of 3rd party plugins it should be $$__local__$$); for example, com.xebialabs.deployit.plugins
     * @param artifactId   the name of artifact on the Nexus repository; for example, xld-docker-plugin
     * @param version      the version of the artifact which should be installed; for example, 10.1.0
     * @return empty in case of successful cancellation or error message in case of failure
     * @restDetails Returns an empty response with a status code of 200 (OK). In case if plugin with the provided data
     * does not exist in the system returns a status code of 404 (NOT FOUND). In case of an uninstall API call for a
     * plugin in INSTALLED status, returns a status code of 501 (NOT IMPLEMENTED).
     * @apiDetails Returns an empty response with a status code of 200 (OK). In case if plugin with the provided data
     * does not exist in the system returns a status code of 404 (NOT FOUND). In case of an uninstall API call for a
     * plugin in INSTALLED status, returns a status code of 501 (NOT IMPLEMENTED).
     * @permission Available only to admin
     */
    @DELETE
    @Path("/delete/{repositoryId}/{groupId}/{artifactId}{version:(/.*)?}")
    Response uninstall(@PathParam("repositoryId") String repositoryId,
                       @PathParam("groupId") String groupId,
                       @PathParam("artifactId") String artifactId,
                       @PathParam("version") String version);

    /**
     * The search is performed amongst all available plugins regardless of their installation status.
     * It's performed against the plugin id, which is a string in the following format: `repository-id:groupId:artifactId-version.extension`
     * Search endpoint will try to find a plugin whose plugin id contains the provided `query` path parameter (no wildcards).
     *
     * @param query the search query to be used for plugin lookup
     * @return a list of found plugins
     * @restDetails Returns a response with a status code of 200 (OK) and a list of found plugins with all details.
     * @apiDetails Returns a response with a status code of 200 (OK) and a list of found plugins with all details.
     * @permission Available only to admin
     */
    @GET
    @Path("search/{query}")
    @Produces({MediaType.APPLICATION_JSON})
    Response search(@PathParam("query") String query);

    /**
     * Updates cached list of available official plugins in the system for a given repository by downloading the metadata from the Nexus repository.
     * Sometimes plugins are updated and released to Nexus so it is good to periodically do this API call in order to update the cached list of plugins in the system.
     *
     * The response contains date when repository metadata was downloaded from Nexus last time.
     *
     * @param repositoryId identifier of  the Nexus repository; for example, xld-official
     * @return timestamp of the update
     * @restDetails Returns a response with a status code of 200 (OK) and a timestamp of the update.
     * @apiDetails Returns a response with a status code of 200 (OK) and a timestamp of the update.
     * @permission Available only to admin
     */
    @POST
    @Path("repositories/{repositoryId}/update")
    Response updateRepository(@PathParam("repositoryId") String repositoryId);

    /**
     * Gets the plugin logo.
     *
     * @param repositoryId the Nexus repository ID on which plugin is accessible (in case of 3rd party plugins it should be $$__local__$$); for example, xld-official
     * @param groupId      the coordinates of plugin on the Nexus repository (in case of 3rd party plugins it should be $$__local__$$); for example, com.xebialabs.deployit.plugins
     * @param artifactId   the name of artifact on the Nexus repository; for example, xld-docker-plugin
     * @return image file
     * @restDetails Returns an response with a status code of 200 (OK) and the image file.
     * @apiDetails Returns an response with a status code of 200 (OK) and the image file.
     * @permission Available only to admin
     */
    @GET
    @Path("repositories/{repositoryId}/logo/{groupId}/{artifactId}")
    @Produces({"image/*"})
    Response getLogo(@PathParam("repositoryId") String repositoryId,
                     @PathParam("groupId") String groupId,
                     @PathParam("artifactId") String artifactId);

}
